import numpy as np
import pandas as pd
import matplotlib.pyplot as plt

def create_unsupervised_summary_table(model_results, metrics):
    print("\n" + "="*80)
    print("UNSUPERVISED EVALUATION RESULTS")
    print("="*80)
    
    for model_name in model_results.keys():
        conf_mean = metrics[model_name]['confidence_stats']['mean']
        conf_var = metrics[model_name]['confidence_stats']['variance']
        avg_time = metrics[model_name]['avg_inference_time']
        pred_dist = metrics[model_name]['prediction_distribution']
        
        print(f"\n{model_name} Performance:")
        print(f"   • Average Confidence: {conf_mean:.3f}")
        print(f"   • Confidence Variance: {conf_var:.6f}")
        print(f"   • Average Speed: {avg_time:.1f}ms")
        print(f"   • Prediction Distribution: {pred_dist}")
        
        # Add performance indicators
        if conf_mean > 0.95:
            print(f"   High confidence - suitable for production")
        if conf_var < 0.01:
            print(f"   Very consistent predictions")
        if avg_time < 50:
            print(f"   Fast inference - real-time capable")
    
    print("\n" + "="*80)

def plot_unsupervised_results(model_results, metrics):
    print("\nGenerating unsupervised visualizations...")
    
    # Create 3 subplots with better spacing
    fig, axes = plt.subplots(1, 3, figsize=(20, 6))
    
    # Plot 1: Speed vs Confidence scatter (auto-scale)
    for model_name in model_results.keys():
        conf = metrics[model_name]['confidence_stats']['mean']
        speed = metrics[model_name]['avg_inference_time']
        axes[0].scatter(speed, conf, s=150, label=model_name)
    
    axes[0].set_xlabel('Average Inference Time (ms)')
    axes[0].set_ylabel('Average Confidence')
    axes[0].set_title('Speed vs Confidence')
    axes[0].legend()
    axes[0].grid(True, alpha=0.3)
    
    # Plot 2: Confidence Distribution
    for model_name in model_results.keys():
        confidences = model_results[model_name]['confidences']
        axes[1].hist(confidences, bins=15, alpha=0.7, label=model_name, edgecolor='black')
    
    axes[1].set_xlabel('Confidence Score')
    axes[1].set_ylabel('Frequency')
    axes[1].set_title('Confidence Distribution')
    axes[1].legend()
    axes[1].grid(True, alpha=0.3)
    
    # Plot 3: Inference Time Box Plot with Outlier Filtering
    inference_data = []
    model_labels = []
    outlier_info = []
    
    for model_name in model_results.keys():
        times = np.array(model_results[model_name]['inference_times'])
        
        # Calculate IQR and filter outliers
        Q1 = np.percentile(times, 25)
        Q3 = np.percentile(times, 75)
        IQR = Q3 - Q1
        lower_bound = Q1 - 1.5 * IQR
        upper_bound = Q3 + 1.5 * IQR
        
        # Filter out outliers for visualization
        filtered_times = times[(times >= lower_bound) & (times <= upper_bound)]
        outliers = times[(times < lower_bound) | (times > upper_bound)]
        
        inference_data.append(filtered_times)
        model_labels.append(model_name)
        outlier_info.append({
            'count': len(outliers),
            'max': np.max(outliers) if len(outliers) > 0 else 0,
            'filtered_count': len(filtered_times),
            'mean': np.mean(filtered_times),
            'median': np.median(filtered_times)
        })
    
    # Create box plot with filtered data
    box_plot = axes[2].boxplot(inference_data, tick_labels=model_labels, patch_artist=True)
    
    # Make box plot more visible
    for patch in box_plot['boxes']:
        patch.set_facecolor('lightblue')
        patch.set_alpha(0.7)
    
    # Add statistical annotations
    for i, (model_name, info) in enumerate(zip(model_labels, outlier_info)):
        # Add mean and median annotations
        axes[2].text(i+1, info['mean'], f'μ={info["mean"]:.1f}ms', 
                    ha='center', va='bottom', fontsize=8, 
                    bbox=dict(boxstyle='round,pad=0.2', facecolor='yellow', alpha=0.7))
        
        # Add outlier information if any
        if info['count'] > 0:
            axes[2].text(i+1, axes[2].get_ylim()[1] * 0.9, 
                        f'{info["count"]} outliers\n(max: {info["max"]:.1f}ms)', 
                        ha='center', va='top', fontsize=7,
                        bbox=dict(boxstyle='round,pad=0.2', facecolor='orange', alpha=0.7))
    
    axes[2].set_ylabel('Inference Time (ms)')
    axes[2].set_title('Inference Time Distribution (Outliers Filtered)')
    axes[2].grid(True, alpha=0.3)
    
    plt.tight_layout()
    plt.show()
    
    # Add explanations BELOW the graphs
    print("\n" + "="*80)
    print("HOW TO INTERPRET THE RESULTS")
    print("="*80)
    
    print("\nSpeed vs Confidence (Left Graph):")
    print("   • Shows the correlation between speed and confidence")
    print("   • Each point represents a model's average performance")
    print("   • Ideal position: High confidence with fast speed")
    
    print("\nConfidence Distribution (Middle Graph):")
    print("   • Shows the confidence of answering the questions")
    print("   • Histogram of how confident the model is across all predictions")
    print("   • Peak near 1.0 = Model is very certain about its answers")
    
    print("\nInference Time Distribution (Right Graph):")
    print("   • Shows how fast the model is across each sample")
    print("   • Box plot of prediction speed for individual samples")
    print("   • Outliers filtered for better visualization of typical performance")
    
    print("\nWhat This Means for Your Model:")
    for model_name in model_results.keys():
        conf_stats = metrics[model_name]['confidence_stats']
        speed_stats = metrics[model_name]['avg_inference_time']
        
        print(f"\n   {model_name} Analysis:")
        print(f"      → Confidence: {conf_stats['mean']:.3f} (variance: {conf_stats['variance']:.6f})")
        print(f"      → Speed: {speed_stats:.1f}ms average")
        
        # Add specific insights based on the actual data
        if conf_stats['variance'] < 0.00001:
            print(f"      Extremely consistent predictions (variance < 0.00001)")
        elif conf_stats['variance'] < 0.01:
            print(f"      Very consistent predictions")
            
        if conf_stats['mean'] > 0.99:
            print(f"      Exceptionally high confidence - excellent for production")
        elif conf_stats['mean'] > 0.95:
            print(f"      High confidence - good for production")
            
        if speed_stats < 20:
            print(f"      Very fast inference - excellent for real-time applications")
        elif speed_stats < 50:
            print(f"      Fast inference - suitable for real-time")
    
    print("="*80)
    axes[2].set_ylabel('Inference Time (ms)')
    axes[2].set_title('Inference Time Distribution')
    axes[2].grid(True, alpha=0.3)
    
    plt.tight_layout()
    plt.show()

def generate_unsupervised_recommendations(model_results, metrics):
    print("\nUnsupervised Evaluation Recommendations:")
    print("-" * 50)
    
    model_names = list(model_results.keys())
    if len(model_names) == 1:
        model_name = model_names[0]
        conf = metrics[model_name]['confidence_stats']['mean']
        speed = metrics[model_name]['avg_inference_time']
        
        if conf > 0.95:
            print(f"{model_name} shows high confidence ({conf:.3f}) - good for production")
        if speed < 50:
            print(f"{model_name} has fast inference ({speed:.1f}ms) - suitable for real-time")
        
        variance = metrics[model_name]['confidence_stats']['variance']
        if variance < 0.01:
            print(f"{model_name} shows consistent predictions (low variance)")
    else:
        print("Multiple model comparison recommendations would appear here")
