function Copy-Entries ($Source, $Target) {
    $Source.GetEnumerator() | ForEach-Object { $_.Key } | % {
        $Target[$_] = $Source[$_]
    }
}
function Write-Table ($in) {
    $in | Format-Table -AutoSize -Wrap | Out-String | Write-Host
}
function Convert-CIDRToIpRange {
<#
.SYNOPSIS
    Convert CIDR notation to start and end of IP range

.DESCRIPTION
    Convert CIDR notation to IP range values to enable checking
    for range containment and overlaps

.EXAMPLE
    Convert-CIDRToIpRange '10.0.0.0/8'

.LINK
    See CIDR prefix notation definition in https://tools.ietf.org/html/rfc4632
#>
[CmdletBinding()]
[OutputType([uint32[]])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [string]
    $CIDR # CIDR
)
Process {
    # RFC 4632 states
    # "In CIDR notation, a prefix is shown as a 4-octet
    # quantity, just like a traditional IPv4 address or network number,
    # followed by the "/" (slash) character, followed by a decimal value
    # between 0 and 32 that describes the number of significant bits."
    $CIDRComponents = $CIDR.Split("/")
    If (2 -ne $CIDRComponents.Length) {
        $message = "CIDR must contain an IP v4 address and mask bit length separated by '/'! e.g. 10.0.0.0/8. Given $CIDR."
        Write-Error $message
        throw $message
    }

    $octets = [uint32[]]($CIDRComponents[0].Split('.'))
    If (4 -ne $octets.Length) {
        $message = "An IPv4 address must consist of 4 octets! e.g. 10.0.0.0. Given $CIDRComponents[0]"
        Write-Error $message
        throw $message
    }
    $ipNum = ($octets[0] -shl 24) + ($octets[1] -shl 16) + ($octets[2] -shl 8) + $octets[3]

    [int32]$maskbits = $CIDRComponents[1]
    [int32]$mask = 0xffffffff
    $mask = $mask -shl (32 - $maskbits)

    $ipStart = $ipNum -band $mask;
    $ipEnd = $ipNum -bor ($mask -bxor 0xffffffff)

    Return ($ipStart, $ipEnd)
}
}

function Assert-CIDRContains {
<#
.SYNOPSIS
    Assert that one CIDR range contains another

.DESCRIPTION
    Assert that parent CIDR completely contains child CIDR

.EXAMPLE
    Assert-CIDRContains -ParentCIDR '10.0.0.0/8' -ChildCIDR '10.3.0.0/16'

.EXAMPLE
    Assert-CIDRContains '10.0.0.0/8' '10.3.0.0/16'

.LINK
    See CIDR prefix notation definition in https://tools.ietf.org/html/rfc4632
#>
[CmdletBinding()]
[OutputType([boolean])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [string]
    $ParentCIDR,

    [Parameter(Mandatory = $True,
               Position = 1)]
    [string]
    $ChildCIDR
)
Process {
    $ParentStart, $ParentEnd = Convert-CIDRToIpRange($ParentCIDR)
    $ChildStart, $ChildEnd = Convert-CIDRToIpRange($ChildCIDR)

    Return (($ChildStart -le $ChildEnd) -and
     ($ParentStart -le $ParentEnd) -and
     ($ParentStart -le $ChildStart) -and
     ($ChildEnd -le $ParentEnd))
}
}

function Assert-IpIsFSxAccessible {
<#
.SYNOPSIS
    Assert that a given IP is accessible by FSx

.DESCRIPTION
    Assert that a given IP is accessible by Amazon FSx.
    Per https://docs.aws.amazon.com/fsx/latest/WindowsGuide/supported-fsx-clients.html#access-environments,
    this is any address which does not conflict with 198.19.0.0/16 or any Amazon-owned CIDR range.

.EXAMPLE
    Assert-IpIsFSxAccessible -Ip '10.0.1.128' -VpcCIDR '11.0.0.0/8'

.EXAMPLE
    Assert-IpIsFSxAccessible '10.0.1.128' '11.0.0.0/8'
#>
[CmdletBinding()]
[OutputType([boolean])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [string]
    $Ip,

    [Parameter(Mandatory = $True,
               Position = 1)]
    [string]
    $VpcCIDR
)
Process {
    $WebClient = New-Object System.Net.WebClient
    $CIDRs = (($WebClient.DownloadString("https://ip-ranges.amazonaws.com/ip-ranges.json") | ConvertFrom-Json).prefixes |
        Where-Object {$_.service -eq "AMAZON"} |
        Where-Object {$_.ip_prefix -ne $null} |
        ForEach-Object {$_.ip_prefix}) + '198.19.0.0/16'
    ForEach ($CIDR in $CIDRs) {
        If (Assert-CIDRContains -ParentCIDR $CIDR -ChildCIDR "${Ip}/32") {
            Return $False
        }
    }
    Return $True
}
}
function Test-FSxADTcpPort {
<#
.SYNOPSIS
    Test a TCP port from Amazon FSx to Active Directory server

.DESCRIPTION
    Test TCP connectivity from Amazon FSx file server to an Active Directory
    server that said file server is attempting to integrate
    with. This helps debug connectivity issues such as security groups or
    firewalls.

.EXAMPLE
    Test-FSxADTcpPort -Server '10.0.0.1' -Port 53

.EXAMPLE
    Test-FSxADTcpPort -Server '10.0.0.1' -Port 53 -Timeout 3000

.LINK
    For more detailed debugging on Microsoft platforms, consider using PortQuery

    Microsoft blog for testing domain controller connectivity
    https://blogs.msdn.microsoft.com/muaddib/2009/03/29/testing-domain-controller-connectivity-using-portqry

    Port query description
    https://support.microsoft.com/en-us/help/310099/description-of-the-portqry-exe-command-line-utility

    Port query v2 download
    https://www.microsoft.com/en-us/download/details.aspx?id=17148
#>
[CmdletBinding()]
[OutputType([string])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [string]
    $Server, # IP v4 address of the AD server


    [Parameter(Mandatory = $True,
               Position = 1)]
    [int]
    $Port, # TCP port number

    [Parameter(Mandatory = $False,
               Position = 2)]
    [ValidateRange(1, 15000)]
    [int]
    $Timeout = 3000 # Timeout in milliseconds
)
Process {
    $Result = "Unknown"
    $TcpClient = New-Object System.Net.Sockets.TcpClient
    $TcpClient.ReceiveTimeout = $Timeout
    Try {
        $Connect = $TcpClient.BeginConnect($Server, $Port, $null, $null)
        $Wait = $Connect.AsyncWaitHandle.WaitOne($Timeout, $false)
        if ($Wait) {
            $Result = "Listening"
        } else {
            $Result = "Timed Out"
        }
    } Catch [System.Net.Sockets.SocketException] {
        $SocketException = $_.Exception
        $Result = "SocketError $SocketException.ErrorCode"
    } Finally {
        $TcpClient.Dispose()
    }

    Return $Result
}
}
function Test-FSxADUdpPort {
<#
.SYNOPSIS
    Test a UDP port from Amazon FSx to Active Directory server

.DESCRIPTION
    Test UDP connectivity from Amazon FSx file server to an Active Directory
    server that said file server is attempting to integrate
    with. This helps debug connectivity issues such as security groups or
    firewalls.

.EXAMPLE
    Test-FSxADUdpPort -Server '10.0.0.1' -Port 53

.EXAMPLE
    Test-FSxADUdpPort -Server '10.0.0.1' -Port 53 -Timeout 3000

.LINK
    For more detailed debugging on Microsoft platforms, consider using PortQuery

    Microsoft blog for testing domain controller connectivity
    https://blogs.msdn.microsoft.com/muaddib/2009/03/29/testing-domain-controller-connectivity-using-portqry

    Port query description
    https://support.microsoft.com/en-us/help/310099/description-of-the-portqry-exe-command-line-utility

    Port query v2 download
    https://www.microsoft.com/en-us/download/details.aspx?id=17148
#>
[CmdletBinding()]
[OutputType([string])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [string]
    $Server, # IP v4 address of the AD server


    [Parameter(Mandatory = $True,
               Position = 1)]
    [int]
    $Port, # UDP port number

    [Parameter(Mandatory = $False,
               Position = 2)]
    [ValidateRange(1, 15000)]
    [int]
    $Timeout = 3000 # Timeout in milliseconds
)
Process {
    $Result = "Unknown"
    $UdpClient = New-Object System.Net.Sockets.UdpClient
    $UdpClient.Client.ReceiveTimeout = $Timeout
    Try {
        $Connect = $UdpClient.Connect($Server, $Port)
        $Acsii = New-Object System.Text.ASCIIEncoding
        $Bytes = $Acsii.GetBytes("?");
        [void]$UdpClient.Send($Bytes, $Bytes.length)
        $RemoteEndpoint = New-Object System.Net.IpEndpoint([System.Net.IpAddress]::Any, 0)
        $ReceiveBytes = $UdpClient.Receive([ref]$RemoteEndpoint)
        [string]$ReturnData = $Acsii.GetString($ReceiveBytes)
        If ($ReturnData) {
            $Result = "Listening"
        }
    } Catch [System.Net.Sockets.SocketException] {
        $SocketException = $_.Exception
        # https://msdn.microsoft.com/en-us/library/ms740668.aspx
        If (($SocketException.ErrorCode -eq 10054) -or ($SocketException.ErrorCode -eq 10061)) {
            $Result = "SocketError $SocketException.ErrorCode"
        } Else {
            # Treat everything other than explicitly refused to be success for UDP.
            $Result = "Listening or Filtered"
        }
    } Finally {
        $UdpClient.Dispose()
    }

    Return $Result
}
}

function Test-FSxADConnection {
<#
.SYNOPSIS
    Test network from Amazon FSx to Active Directory server for required ports

.DESCRIPTION
    Test network from Amazon FSx to Active Directory server for required ports
#>
param (
    [Parameter(Mandatory = $True)]
    [string]
    $Server, # IP v4 address of the server

    [Parameter(Mandatory = $False)]
    [ValidateRange(1, 15000)]
    [int]
    $TcpTimeout = 3000, # TCP Timeout in milliseconds

    [Parameter(Mandatory = $False)]
    [ValidateRange(1, 15000)]
    [int]
    $UdpTimeout = 3000, # UDP Timeout in milliseconds

    [Parameter(Mandatory = $True)]
    [string]
    $RecommendedAction, # Recommended action in event of connectivity failure

    [Parameter(Mandatory = $True)]
    [int[]]
    $RequiredTcpPorts, # Required TCP ports

    [Parameter(Mandatory = $True)]
    [int[]]
    $RequiredUdpPorts, # Required UDP ports

    [Parameter(Mandatory = $True)]
    [hashtable]
    $PortDescriptions # Map of port number to description
)
Process {
    $CheckPrerequisitesGuide = "Check pre-requisites in https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-manage-prereqs.html"
    $TotalChecks = $RequiredTcpPorts.Count + $RequiredUdpPorts.Count

    $ChecksRan = 0
    $TestReport = @{
        "Server" = $Server
        "TcpDetails" = @();
        "UdpDetails" = @()
    }
    $FailedTcpPorts = @()
    ForEach ($TcpPort in $RequiredTcpPorts) {
        $TcpResult = Test-FSxADTcpPort -Server $Server -Port $TcpPort -Timeout $TcpTimeout
        $Description = $PortDescriptions[$TcpPort]
        $Operation = "TCP $TcpPort - $Description"
        $TestReport["TcpDetails"] += New-Object -TypeName PSObject -Property @{
            "Result" = $TcpResult;
            "Port" = $TcpPort;
            "Description" = $Description
        }
        $ChecksRan += 1
        If ($TcpResult -ne "Listening") {
            $FailedTcpPorts += $TcpPort
            Write-Warning -Message "TCP $TcpPort failed to connect. Required for $Description. $RecommendedAction"
        }
        Write-Progress -Activity "Test connection" -Status $Server -PercentComplete ($ChecksRan * 100 / $TotalChecks) -CurrentOperation $Operation
    }
    If ($FailedTcpPorts.Count -gt 0) {
        $TestReport["FailedTcpPorts"] = $FailedTcpPorts
    }
    $FailedUdpPorts = @()
    ForEach ($UdpPort in $RequiredUdpPorts) {
        $UdpResult = Test-FSxADUdpPort -Server $Server -Port $UdpPort -Timeout $UdpTimeout
        $Description = $PortDescriptions[$UdpPort]
        $Operation = "UDP $UdpPort - $Description"
        $TestReport["UdpDetails"] += New-Object -TypeName PSObject -Property @{
            "Result" = $UdpResult;
            "Port" = $UdpPort;
            "Description" = $Description
        }
        $ChecksRan += 1
        If (-not $UdpResult.Contains("Listening")) {
            $FailedUdpPorts += $UdpPort
            Write-Warning -Message "UDP $UdpPort failed to connect. Required for $Description. $RecommendedAction"
        }
        Write-Progress -Activity "Test connection" -Status $Server -PercentComplete ($ChecksRan * 100 / $TotalChecks) -CurrentOperation $Operation
    }
    If ($FailedUdpPorts.Count -gt 0) {
        $TestReport["FailedUdpPorts"] = $FailedUdpPorts
    }
    $FailedPortCount = $FailedTcpPorts.Count + $FailedUdpPorts.Count
    $Success = $FailedPortCount -eq 0
    If ($Success) {
        Write-Information "Connection successful to $Server"
    } Else {
        Write-Warning -Message "$FailedPortCount ports failed to connect to $Server. $CheckPrerequisitesGuide"
    }
    $TestReport["Success"] = $Success

    Return $TestReport
}
}

function Test-FSxADControllerConnection {
<#
.SYNOPSIS
    Test network from Amazon FSx to domain controller

.DESCRIPTION
    Test network from Amazon FSx to domain controller

    Verifies that the required port connectivity as specified in
    https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-manage-prereqs.html
    is available

    As of 2020-06-03
    UDP 88, 123, 389, 464
    TCP 88, 135, 389, 445, 464, 636, 3268, 3269, 9389, 49152-65535

.EXAMPLE
    Test-FSxADControllerConnection -ADControllerIp '10.0.0.1'

.EXAMPLE
    @('10.0.0.1', '10.0.0.2') | Test-FSxADControllerConnection

.EXAMPLE
    Test-FSxADControllerConnection -ADControllerIp '10.0.0.1' -TcpTimeout 3000 -UdpTimeout 5000

.OUTPUTS
    Custom object with following fields
        Success - boolean value on whether test was successful
        Server - IP v4 of domain controller
        TcpDetails - details for each TCP port tested. Result, Port, Description
        FailedTcpPorts - list of failed TCP ports. Only populated if any failed
        UdpDetails - details for each UDP port tested. Result, Port, Description
        FailedUdpPorts - list of failed UDP ports. Only populated if any failed

.LINK
    Port connectivity required for Amazon FSx for Windows Active Directory integration
    https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-manage-prereqs.html

    For more detailed debugging on Microsoft platforms, consider using PortQuery

    Microsoft blog for testing domain controller connectivity
    https://blogs.msdn.microsoft.com/muaddib/2009/03/29/testing-domain-controller-connectivity-using-portqry

    Port query description
    https://support.microsoft.com/en-us/help/310099/description-of-the-portqry-exe-command-line-utility

    Port query v2 download
    https://www.microsoft.com/en-us/download/details.aspx?id=17148
#>
[CmdletBinding()]
[OutputType([PSObject])]
param (
    [Parameter(Mandatory = $True,
               ValueFromPipeline = $True,
               Position = 0)]
    [string]
    $ADControllerIp, # IP v4 address of the domain controller

    [Parameter(Mandatory = $False)]
    [ValidateRange(1, 15000)]
    [int]
    $TcpTimeout = 3000, # TCP Timeout in milliseconds

    [Parameter(Mandatory = $False)]
    [ValidateRange(1, 15000)]
    [int]
    $UdpTimeout = 3000 # UDP Timeout in milliseconds
)
Begin {
    $RecommendedAction = "Verify security group and firewall settings on both client and domain controller."
    $RequiredTcpPorts = @(88, 135, 389, 445, 464, 636, 3268, 3269, 9389)
    $RequiredUdpPorts = @(88, 123, 389, 464)
    $PortDescriptions = @{
        88 = "Kerberos authentication";
        123 = "Network Time Protocol (NTP)";
        135 = "DCE / EPMAP (End Point Mapper)";
        389 = "Lightweight Directory Access Protocol (LDAP)";
        445 = "Directory Services SMB file sharing";
        464 = "Kerberos Change/Set password";
        636 = "Lightweight Directory Access Protocol over TLS/SSL (LDAPS)";
        3268 = "Microsoft Global Catalog";
        3269 = "Microsoft Global Catalog over SSL";
        9389 = "Microsoft AD DS Web Services, PowerShell"
    }
}
Process {
    Return (Test-FSxADConnection -Server $ADControllerIp -TcpTimeout $TcpTimeout -UdpTimeout $UdpTimeout -RecommendedAction $RecommendedAction -RequiredTcpPorts $RequiredTcpPorts -RequiredUdpPorts $RequiredUdpPorts -PortDescriptions $PortDescriptions)
}
}

function Test-FSxADDnsConnection {
<#
.SYNOPSIS
    Test network from Amazon FSx to DNS server

.DESCRIPTION
    Test network from Amazon FSx to DNS server

    Verifies that the required port connectivity as specified in
    https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-manage-prereqs.html
    is available

    UDP 53
    TCP 53

.EXAMPLE
    Test-FSxADDnsConnection -ADDnsIp '10.0.0.1'

.EXAMPLE
    @('10.0.0.1', '10.0.0.2') | Test-FSxADDnsConnection

.EXAMPLE
    Test-FSxADDnsConnection -ADDnsIp '10.0.0.1' -TcpTimeout 3000 -UdpTimeout 5000

.OUTPUTS
    Custom object with following fields
        Success - boolean value on whether test was successful
        Server - IP v4 of DNS server
        TcpDetails - details for each TCP port tested. Result, Port, Description
        FailedTcpPorts - list of failed TCP ports. Only populated if any failed
        UdpDetails - details for each UDP port tested. Result, Port, Description
        FailedUdpPorts - list of failed UDP ports. Only populated if any failed
#>
[CmdletBinding()]
[OutputType([PSObject])]
param (
    [Parameter(Mandatory = $True,
               ValueFromPipeline = $True,
               Position = 0)]
    [string]
    $ADDnsIp, # IP v4 address of the DNS server

    [Parameter(Mandatory = $False)]
    [ValidateRange(1, 15000)]
    [int]
    $TcpTimeout = 3000, # TCP Timeout in milliseconds

    [Parameter(Mandatory = $False)]
    [ValidateRange(1, 15000)]
    [int]
    $UdpTimeout = 3000 # UDP Timeout in milliseconds
)
Begin {
    $RecommendedAction = "Verify security group and firewall settings on both client and DNS server."
    $RequiredTcpPorts = @(53)
    $RequiredUdpPorts = @(53)
    $PortDescriptions = @{
        53 = "Domain Name System (DNS)"
    }
}
Process {
    Return (Test-FSxADConnection -Server $ADDnsIp -TcpTimeout $TcpTimeout -UdpTimeout $UdpTimeout -RecommendedAction $RecommendedAction -RequiredTcpPorts $RequiredTcpPorts -RequiredUdpPorts $RequiredUdpPorts -PortDescriptions $PortDescriptions)
}
}

function Resolve-FSxDcDnsRecord {
<#
.SYNOPSIS
    Resolve DC DNS records via supplied DNS servers to return domain controllers

.DESCRIPTION
    Resolve DC DNS records via supplied DNS servers to return domain controllers

.EXAMPLE
    Resolve-FSxDcDnsRecord -SrvRecord "_ldap._tcp.dc._msdcs.test-ad.local" -DnsIpAddresses @("10.0.88.17", "10.0.77.202") -VpcCIDR "10.0.0.0/16"

.LINK
    See description of process in https://blogs.msmvps.com/acefekay/category/dc-locator-process

    SRV record https://tools.ietf.org/html/rfc2782
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True)]
    [string]
    $SrvRecord,

    [Parameter(Mandatory = $True)]
    [string[]]
    $DnsIpAddresses,

    [Parameter(Mandatory = $True)]
    [string]
    $VpcCIDR
)
Process {
    $Failures = @{}
    $Warnings = @{}
    $DomainControllers = New-Object System.collections.hashtable
    $DnsPassed = $False
    $DnsConnected = $False
    $DnsCommunication = "DnsCommunication"
    $FailingDnsIps = @()
    ForEach ($Ip in $DnsIpAddresses) {
        $Error.Clear()
        If (Assert-IpIsFSxAccessible -Ip $Ip -VpcCIDR $VpcCIDR) {
            $Results = (Resolve-DnsName -Name $SrvRecord -Server $Ip -Type SRV -QuickTimeout -ErrorAction SilentlyContinue)
            If (!$Error) {
                $DnsPassed = $True
                $NameToAResult = New-Object System.collections.hashtable
                $NameToSrvResult = New-Object System.collections.hashtable
                ForEach ($Result in $Results) {
                    If (($Result.Type -eq "SRV") -and (-not ($DomainControllers.Contains($Result.NameTarget)))) {
                        $NameToSrvResult[$Result.NameTarget] = ($Result | Select Priority, Weight)
                    } ElseIf (($Result.Type -eq "A") -and (-not ($DomainControllers.Contains($Result.Name)))) {
                        $NameToAResult[$Result.Name] = ($Result | Select Name, IPAddress)
                    }
                }
                ForEach ($DCName in $NameToSrvResult.Keys) {
                    $SrvResult = $NameToSrvResult[$DCName]
                    $AResult = $NameToAResult[$DCName]
		    If (-not ([string]::IsNullOrEmpty($AResult.IPAddress))) {
                    $DomainControllers[$DCName] = [PSCustomObject]@{
                        Name = $AResult.Name
                        IPAddress = $AResult.IPAddress
                        Priority = $SrvResult.Priority
                        Weight = $SrvResult.Weight
                    }
                }

	     }
            } Else {
                If ($Error[0].Exception.Message -match " timeout ") {
                    Write-Warning "Unable to communicate with the following DNS Server: ${Ip}"
                } Else {
                    Write-Warning "Unable to resolve ${SrvRecord} using the following DNS Server: ${Ip}"
                    $DnsConnected = $True
                }
            }
        } Else {
            Write-Warning "Unable to communicate with the following DNS Server because its IP is outside the file system's VPC and is not an RFC1918 IP address: ${Ip}"
            $FailingDnsIps += $Ip
            $Warnings[$DnsCommunication] = $FailingDnsIps
        }
    }
    $OrderedDomainControllers = @()
    If ($DnsPassed) {
        $DCCount = $DomainControllers.Count
        If ($DCCount -gt 0) {
            # Per https://tools.ietf.org/html/rfc2782,
            # A client MUST attempt to contact the target host with the lowest-numbered priority it can reach;
            # target hosts with the same priority SHOULD be tried in an order defined by the weight field.
            # Larger weights SHOULD be given a proportionately higher probability of being selected.
            $OrderedDomainControllers = ($DomainControllers.Values | Sort-Object -Property @{Expression = "Priority"; Descending = $False}, @{Expression = "Weight"; Descending = $True})
            Write-Host "`nFound the following $DCCount Domain Controllers by querying ${SrvRecord}: "
            Write-Table $OrderedDomainControllers
        } Else {
            Write-Warning "No domain controllers found for ${SrvRecord}!"
            $Failures.Add("InvalidDomain", $SrvRecord)
        }
    } Else {
        If ($DnsConnected) {
            Write-Warning "Unable to resolve ${SrvRecord} using any provided DNS server!"
            $Failures.Add("DnsResolution", $SrvRecord)
        } Else {
            $Failures.Add($DnsCommunication, $DnsIpAddresses)
            # No point in returning same error in both warning and error
            $Warnings.Remove($DnsCommunication)
        }
    }

    Return @{
        DomainControllers = $OrderedDomainControllers
        Failures = $Failures
        Warnings = $Warnings
    }
}
}

function Test-Subnets {
<#
.SYNOPSIS
    Test subnets match criteria for FSx integration

.DESCRIPTION
    Test subnets match criteria for FSx integration

.EXAMPLE
    $SubnetIds = @('subnet-04431191671ac0d19', 'subnet-0f30db1cad3a599d1')
    $Context = @{
        SubnetIds = $SubnetIds
        Failures = @{}
        Skip = 0
        Test = 1
        TestHeaderColor = "Green"
    }
    Test-Subnets -Context $Context

.LINK
    See CIDR prefix notation definition in https://tools.ietf.org/html/rfc4632
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate EC2 Subnets"

    If ($Context.Failures.Count -gt 0) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    If (($Context.SubnetIds.Length -eq 0) -or ($Context.SubnetIds.Length -gt 2)) {
        Write-Warning "Must pass either one or two subnet IDs!"
        $Context.Failures.Add("InvalidSubnetCount", $Context.SubnetIds.Length)
        $Context.Test++
        Return $Context
    }

    Try {
        $Region = (Get-EC2InstanceMetadata -Category Region).SystemName
        Write-Host "Found region $Region from EC2 instance metadata"
    } Catch {
        $DefaultRegion = Get-DefaultAWSRegion
        If ($DefaultRegion) {
            $Region = $DefaultRegion.Region
            Write-Host "Falling back to shell default region $Region"
        } Else {
            $Region = "us-east-1"
            Write-Host "No region found in either EC2 instance metadata or shell default, falling back to global default $Region"
            Write-Host "If you would like to use another region, use Set-DefaultAWSRegion and re-run"
        }
    }

    Try {
        $Subnets = (Get-Ec2Subnet -SubnetId $Context.SubnetIds -Region $Region)
    } Catch {
        $ErrorMessage = $_.Exception.Message
        If ($ErrorMessage -match "does not exist") {
            # The subnet ID 'subnet-07456164' does not exist
            Write-Warning($ErrorMessage)
            $Context.Failures.Add("InvalidSubnetIds", $Context.SubnetIds)
        } Else {
            Write-Warning("Failed to get EC2 subnets due to " + $_)
            Write-Warning "Please validate instance has 'ec2:DescribeSubnets' permission."
            $Context.Failures.Add("GetEc2Subnet", $Context.SubnetIds)
        }
        $Context.Test++
        Return $Context
    }

    $Context.Add("Subnets", $Subnets)

    Write-Table ($Subnets | Select SubnetId, VpcId, AvailabilityZoneId, CidrBlock)

    If (($Subnets.VpcId | Select -Unique).Count -gt 1) {
        Write-Warning "Subnets must belong to the same VPC!"
        $Context.Failures.Add("SubnetsInSeparateVPCs", @($Subnets[0].VpcId, $Subnets[1].VpcId))
    }

    If ($SubnetIds.Count -ne ($Subnets.AvailabilityZoneId | Select -Unique).Count) {
        Write-Warning "Subnets must belong to different AZs!"
        $Context.Failures.Add("SubnetsInSameAZ", ($Subnets.AvailabilityZoneId))
    }

    Try {
        $VPC = (Get-Ec2Vpc -VpcId $Subnets[0].VpcId -Region $Region)
    } Catch {
        Write-Warning("Failed to get VPC due to " + $_)
        Write-Warning "Please make sure instance has 'ec2:DescribeVpcs' permission."
        $Context.Failures.Add("GetEc2Vpc", $Subnets[0].VpcId)
        $Context.Test++
        Return $Context
    }

    $Context.Add("VPC", $VPC)
    $Context.Test++

    Return $Context
}
}

function Test-Dns {
<#
.SYNOPSIS
    Test DNS resolves properly for FSx integration

.DESCRIPTION
    Test DNS resolves properly for FSx integration

    Can also be used to resolve site-specific domain controllers

.EXAMPLE
    $DnsIpAddresses = @("10.0.88.17", "10.0.77.202")
    $Domain = "test-ad.local"
    $VPC = [PSCustomObject]@{
                CidrBlock = "10.0.0.0/16"
                VpcId = "vpc-00811ed2298428798"
            }
    $FSxAdSiteName = $Null # After site resolution, this also can be 'Site1' etc.
    $Context = @{
        DnsIpAddresses = $DnsIpAddresses
        Domain = $Domain
        FSxAdSiteName = $FSxAdSiteName
        VPC = $VPC
        Warnings = @{}
        Failures = @{}
        Skip = 0
        Test = 2
        TestHeaderColor = "Green"
    }
    Test-Dns -Context $Context

.LINK
    See description of process in https://blogs.msmvps.com/acefekay/category/dc-locator-process
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    If ($Context.FSxAdSiteName) {
        $TestName = "Looking up DNS entries for domain controllers in site " + $Context.FSxAdSiteName
    } Else {
        $TestName = "Validate connectivity with DNS Servers"
    }

    If ($Context.Failures.Count -gt 0) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    If (($Context.DnsIpAddresses.Count -eq 0) -or ($Context.DnsIpAddresses.Count -gt 2)) {
        Write-Warning "Must pass either one or preferably two unique DNS IP addresses!"
        $Context.Failures.Add("InvalidDnsIpCount", $Context.DnsIpAddresses.Count)
        $Context.Test++
        Return $Context
    }

    $SrvRecordPrefix = "_ldap._tcp.dc._msdcs."

    If ($Context.FSxAdSiteName) {
        $SrvRecord = ("_ldap._tcp." + $Context.FSxAdSiteName + "._sites.dc._msdcs." + $Context.Domain)
    } Else {
        $SrvRecord = $SrvRecordPrefix + $Context.Domain
    }

    $Result = (Resolve-FSxDcDnsRecord -SrvRecord $SrvRecord -DnsIpAddresses $Context.DnsIpAddresses -VpcCIDR $Context.VPC.CidrBlock)

    If ($Result.Failures.Count -gt 0) {
        Copy-Entries $Result.Failures $Context.Failures

        If ($Context.FSxAdSiteName) {
            $ResolutionFailedMessage = "Failed to resolve DNS record for site " + $Context.FSxAdSiteName
        } Else {
            $ResolutionFailedMessage = "Failed to resolve DNS record for domain " + $Context.Domain
        }
        Write-Warning $ResolutionFailedMessage

        # A point of confusion is passing a domain controller's DNS name instead of domain name.
        # Check specifically for that case but cap to one level.

        $First, $Rest = $Context.Domain.Split(".")
        If (($Rest -eq $Null) -or $Context.FSxAdSiteName) {
            # No-op
        } Else {
            If ($Rest -Is [string]) {
                $InferredDomainName = $Rest
            } Else  {
                $InferredDomainName = [string]::Join(".", $Rest)
            }
            $InferredSrvRecord = $SrvRecordPrefix + $InferredDomainName
            Write-Host "Trying variation $InferredSrvRecord"
            $InferredResult = (Resolve-FSxDcDnsRecord -SrvRecord $InferredSrvRecord -DnsIpAddresses $Context.DnsIpAddresses -VpcCIDR $Context.VPC.CidrBlock)
            If ($InferredResult.Failures.Count -eq 0) {
                $Domain = $Context.Domain
                Write-Warning "Did you mean to pass in $InferredDomainName instead of ${Domain} for DomainDNSRoot?"
                $Context.Failures.Add("InferredDomainName", $InferredDomainName)
            }
        }
    } Else {
        $Context.DomainControllers = $Result.DomainControllers
    }

    If ($Result.Warnings.Count -gt 0) {
        Copy-Entries $Result.Warnings $Context.Warnings
    }

    $Context.Test++

    Return $Context
}
}

function Find-FSxReachableDC {
<#
.SYNOPSIS
    Find a domain controller that FSx can reach.

.DESCRIPTION
    Find a domain controller that FSx can reach from list of supplied domain controllers
    Fast exit to return first reachable one in the list and will best-effort warn of
    unreachable ones.

.EXAMPLE
    $DomainControllers = @(
        [PSCustomObject]@{
            Name = 'DC1.test-ad.local'
            IPAddress = '10.0.5.228'
        },
        [PSCustomObject]@{
            Name = 'DC2.test-ad.local'
            IPAddress = '10.0.77.202'
        }
    )
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
    Find-FSxReachableDC -DomainControllers $DomainControllers -VpcCIDR "10.0.0.0/16" -Credential $Credential -ResolveAdUser -RunFullNetworkForReachable
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object[]]
    $DomainControllers,

    [Parameter(Mandatory = $True,
               Position = 1)]
    [string]
    $VpcCIDR,

    [Parameter(Mandatory = $True,
               Position = 2)]
    [System.Management.Automation.PSCredential]
    $Credential,

    [Parameter(Mandatory = $False,
               Position = 3)]
    [Switch]
    $ResolveAdUser,

    [Parameter(Mandatory = $False,
               Position = 4)]
    [Switch]
    $RunFullNetworkForReachable # Run full network validation even if DC was reachable (helpful to detect other port issues.)
)
Process {
    $Failures = @{}
    $Warnings = @{}
    $UnreachableDCs = @()
    $CredentialsValid = $True
    $UsersPermissionValid = $False
    $RanNetworkValidation = $False
    $ReachableDCs = @()
    $InvalidCredentialsCounter = 0
    $Forest = ""

    ForEach ($DC in $DomainControllers) {
        If (Assert-IpIsFSxAccessible -Ip $DC.IPAddress -VpcCIDR $VpcCIDR) {
            Try {
                $CommonAdArgs = @{
                    Server = $DC.IPAddress
                    Credential = $Credential
                    ErrorAction = 'SilentlyContinue'
                }
                $AdDomain = (Get-AdDomain @CommonAdArgs)
                $ReachableDCs += $DC
                $NetworkConnectionSuccessful = $True
                $Forest = $AdDomain.Forest
            }
            Catch [System.Security.Authentication.AuthenticationException] {
                Write-Warning "Invalid credentials provided!"
                $NetworkConnectionSuccessful = $True
                $CredentialsValid = $False
            }
            Catch [Microsoft.ActiveDirectory.Management.ADServerDownException] {
                # Unable to contact the server. This may be because this server does not exist,
                # it is currently down, or it does not have the Active Directory Web Services running.
                Write-Warning($_)
                $NetworkConnectionSuccessful = $False
            }
            Catch {
                Write-Warning("Failed to connect due to unknown exception. " + $_)
                $NetworkConnectionSuccessful = $False
            }
            If ($NetworkConnectionSuccessful -and $CredentialsValid -and $ResolveAdUser) {
                Try {
                    $AdUser = (Get-AdUser $Credential.UserName @CommonAdArgs)
                    $UsersPermissionValid = $True
                } Catch [Microsoft.ActiveDirectory.Management.ADIdentityNotFoundException] {
                    $UsersPermissionValid = $False
                }
            }
            If (!$CredentialsValid) {
                Write-Warning("Credentials are invalid when connecting to " + $DC.Name + " (" + $DC.IPAddress + ")")
                $InvalidCredentialsCounter++
                Continue
            }
            If ($ResolveAdUser -and !$UsersPermissionValid) {
                Write-Warning("Unable to locate AD user when connecting to " + $DC.Name + " (" + $DC.IPAddress + ")")
                Continue
            }
            If ($NetworkConnectionSuccessful) {
                If ($ResolveAdUser) {
                    Continue # Validate the service user credentials for other DCs
                } Else {
                    Break # Skip further checks as we already found a reachable DC
                }
            } Else {
                Write-Warning("Unable to communicate with the following AD Domain Controller: " + $DC.Name + " (" + $DC.IPAddress + ")")
                $UnreachableDCs += $DC
                If ($RanNetworkValidation) {
                    Write-Host "Full network validation already ran for another DC, skipping..."
                } Else {
                    Write-Host "Running full network validation to help pinpoint reachability issues"
                    $NetworkValidationResult = Test-FSxADControllerConnection -ADControllerIp $DC.IPAddress
                    $Warnings.Add('DomainControllerNetworkValidation', $NetworkValidationResult)
                    $RanNetworkValidation = $True
                }
            }
        } Else {
            Write-Warning("Unable to communicate with the following AD Domain Controller because its IP is outside the file system's VPC and is not an RFC1918 IP address: " + $DC.Name + " (" + $DC.IPAddress + ")")
            $UnreachableDCs += $DC
        }
    }

    $FirstReachableDC = $ReachableDCs[0]
    If ($RunFullNetworkForReachable -and $FirstReachableDC) {
        $ReachableDCIp = $FirstReachableDC.IPAddress
        Write-Host "Running full network validation against $ReachableDCIp confirm all ports are open"
        $NetworkValidationResult = Test-FSxADControllerConnection -ADControllerIp $ReachableDCIp
        If ($NetworkValidationResult.Success) {
            Write-Host "All ports on $ReachableDCIp are open"
        } Else {
            Write-Host "Some ports on $ReachableDCIp are closed. Please check connectivity."
            $Warnings.Add('DomainControllerNetworkValidation', $NetworkValidationResult)
        }
    }

    $Results = @{
        Failures = $Failures
        Warnings = $Warnings
    }

    $NoCommunicationWithDCs = "NoCommunicationWithDCs"
    If ($NetworkConnectionSuccessful) {
        If ($UnreachableDCs.Count -gt 0) {
            $Warnings.Add($NoCommunicationWithDCs, $UnreachableDCs)
        }
        If ($CredentialsValid) {
            $Results.Add('AdDomain', $AdDomain)
            $Results.Add('ReachableDC', $FirstReachableDC)
            If ($ResolveAdUser) {
                $Results.Add('AdUser', $AdUser)
            }
        } Else {
            $Failures.Add("InvalidCredentials", $InvalidCredentialsCounter)
        }
        If ($ResolveAdUser -and $CredentialsValid -and !$UsersPermissionValid) {
            Write-Warning("Please validate service account user has 'Read' permission on Users container.")
            $Context.Failures.Add("UnauthorizedReadOnUsersContainer", $AdDomain.UsersContainer)
        }
        $Results.Add('CredentialsValid', $CredentialsValid)
    } Else {
        $Failures.Add($NoCommunicationWithDCs, $DomainControllers)
    }

    If ($Forest) {
        $Results.Add('Forest', $Forest)
    } else {
        Write-Warning("Unable to get the forest name")
    }

    Return $Results
}
}

function Test-Credential {
<#
.SYNOPSIS
    Test FSx service user credential is valid

.DESCRIPTION
    Test FSx service user credential is valid

.EXAMPLE
    $Domain = "test-ad.local"
    $DC1 = [PSCustomObject]@{
            Name = 'DC1.test-ad.local'
            IPAddress = '10.0.5.228'
        }
    $DC2 = [PSCustomObject]@{
            Name = 'DC2.test-ad.local'
            IPAddress = '10.0.77.202'
        }
    $VPC = [PSCustomObject]@{
                CidrBlock = "10.0.0.0/16"
                VpcId = "vpc-00811ed2298428798"
            }
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
    $FSxAdSiteName = $Null # After site resolution, this also can be 'Site1' etc.
    $Context = @{
        DomainControllers = @($DC1, $DC2)
        Domain = $Domain
        FSxAdSiteName = $FSxAdSiteName
        VPC = $VPC
        Credential = $Credential
        Warnings = @{}
        Failures = @{}
        Skip = 0
        Test = 3
        TestHeaderColor = "Green"
    }
    Test-Credential -Context $Context
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    If ($Context.FSxAdSiteName) {
        $TestName = "Validate connectivity with at least one AD Domain Controller in AD site " + $Context.FSxAdSiteName
    } Else {
        $TestName = "Validate FSx service user credentials"
    }

    If ($Context.Failures.Count -gt 0) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    $CommonArgs = @{
        DomainControllers = $Context.DomainControllers
        VpcCIDR = $Context.VPC.CidrBlock
        Credential = $Context.Credential
    }
    If ($Context.FSxAdSiteName) {
        # Once we resolve an AD site, FSx <-> DCs should have full connectivity
        # also no point in resolving service user if we did this earlier against another DC.
        $Result = Find-FSxReachableDC @CommonArgs -RunFullNetworkForReachable
    } Else {
        $Result = Find-FSxReachableDC @CommonArgs -ResolveAdUser
    }

    If ($Result.Failures.Count -gt 0) {
        Copy-Entries $Result.Failures $Context.Failures
        $ReachabilityFailureMessage = "Unable to locate a reachable domain controller for "
        If ($Context.FSxAdSiteName) {
            $ReachabilityFailureMessage += ("FSx AD site " + $Context.FSxAdSiteName)
        } Else {
            $ReachabilityFailureMessage += ("domain " + $Context.Domain)
        }
    } Else {
        If ($Context.FSxAdSiteName) {
            $Context.ReachableSiteDC = $Result.ReachableDC
        } Else {
            $Context.AdDomain = $Result.AdDomain
            Write-Table ($Context.AdDomain | Select Name, DNSRoot, Forest, DistinguishedName, ComputersContainer)

            $Context.AdUser = $Result.AdUser
            Write-Table ($Context.AdUser | Select Name, DistinguishedName, Enabled)

            $Context.ReachableDC = $Result.ReachableDC
            Write-Table ($Context.ReachableDC)
        }
        $Context.Forest = $Result.Forest
    }

    If ($Result.Warnings.Count -gt 0) {
        Copy-Entries $Result.Warnings $Context.Warnings
    }

    $Context.Test++

    Return $Context
}
}

function Test-Domain {
<#
.SYNOPSIS
    Test Active Directory domain properties

.DESCRIPTION
    Test Active Directory domain properties

.EXAMPLE
    $AdDomain = [PSCustomObject]@{
        Name = 'test-ad'
        DNSRoot = 'test-ad.local'
        DomainMode = 'Windows2016Domain'
    }
    $Context = @{
        AdDomain = $AdDomain
        Warnings = @{}
        Failures = @{}
        Skip = 0
        Test = 4
        TestHeaderColor = "Green"
    }
    Test-Credential -Context $Context
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate domain properties"

    If ($Context.Failures.Count -gt 0) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    Write-Table ($Context.AdDomain | Select Name, DNSRoot, DomainMode)
    $DomainMode = $Context.AdDomain.DomainMode
    If ($DomainMode -lt "Windows2008R2Domain") {
        Write-Warning "FSx requires minimum AD domain functionality of Windows 2008 R2, found $DomainMode"
        $Context.Failures.Add("OldDomainFunctionalLevel", $DomainMode)
    }

    $DNSRoot = $Context.AdDomain.DNSRoot
    $NameComponents = $DNSRoot.Split('.')
    If ($NameComponents.Count -lt 2) {
        Write-Warning "FSx does not support Single Label Domains (SLD), found $DNSRoot"
        $Context.Failures.Add("SingleLabelDomain", $DNSRoot)
    }

    $Context.Test++

    Return $Context
}
}

function Test-OrganizationalUnit {
<#
.SYNOPSIS
    Test organizational unit works for FSx integration

.DESCRIPTION
    Test organizational unit works for FSx integration

.EXAMPLE
    $OrganizationalUnit = 'OU=Amazon FSx,OU=test-ad,DC=test-ad,DC=local'
    $AdDomain = [PSCustomObject]@{
        Name = 'test-ad'
        DNSRoot = 'test-ad.local'
        DistinguishedName = 'DC=test-ad,DC=local'
        ComputersContainer = 'CN=Computers,DC=test-ad,DC=local'
    }
    $ReachableDC = [PSCustomObject]@{
                Name = 'DC1.test-ad.local'
                IPAddress = '10.0.5.228'
            }
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
    $Context = @{
        OrganizationalUnit = $OrganizationalUnit
        AdDomain = $AdDomain
        ReachableDC = $ReachableDC
        Credential = $Credential
        Failures = @{}
        Skip = 0
        Test = 5
        TestHeaderColor = "Green"
    }
    Test-OrganizationalUnit -Context $Context

.LINK
    Microsoft Distinguished Name format
    https://docs.microsoft.com/en-us/previous-versions/windows/desktop/ldap/distinguished-names
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate organizational unit"

    If ($Context.Failures.Count -gt 0) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    $CommonArgs = @{
        Server = $Context.ReachableDC.IPAddress
        Credential = $Context.Credential
        ErrorAction = 'SilentlyContinue'
    }

    If (!$Context.OrganizationalUnit) {
        $ComputersContainer = $Context.AdDomain.ComputersContainer
        Write-Host ("No organizational unit supplied, defaulting to domain Computers container " + $ComputersContainer)
        $Context.OrganizationalUnit = $ComputersContainer
        $DefaultComputersName = "CN=Computers," + $Context.AdDomain.DistinguishedName
        If ($Context.OrganizationalUnit -eq $DefaultComputersName) {
            Write-Host "Default CN=Computers container detected."
            $DistinguishedNameFilter = ("DistinguishedName -eq '" + $DefaultComputersName + "'")
            $DefaultComputers = Get-ADObject -Filter $DistinguishedNameFilter @CommonArgs
            If ($DefaultComputers) {
                Write-Table ($DefaultComputers)
                $Context.Add("DefaultComputers", $DefaultComputers)
            } Else {
                Write-Warning("Please validate service account user has 'Read' permission on default Computers container " + $DefaultComputersName)
                $Context.Failures.Add("UnauthorizedReadOnComputersContainer", $DefaultComputersName)
            }
            $Context.Test++
            Return $Context
        }
    }

    $DistinguishedNameFilter = ("DistinguishedName -eq '" + $Context.OrganizationalUnit + "'")
    $OUByDistinguishedName = Get-ADOrganizationalUnit -Filter $DistinguishedNameFilter @CommonArgs
    If ($OUByDistinguishedName) {
        Write-Table ($OUByDistinguishedName)
        $Context.Add("OU", $OUByDistinguishedName)
    } Else {
        Write-Warning ("Failed to locate organizational unit for distinguished name " + $Context.OrganizationalUnit + ". Please verify it's in distinguished name format.")
        $Context.Failures.Add("InvalidOrganizationalUnit", $Context.OrganizationalUnit)

        Write-Host "Attempting name match..."

        $NameFilter = ("Name -like '" + $Context.OrganizationalUnit + "'")

        $OUByName = (Get-ADOrganizationalUnit -Filter $NameFilter @CommonArgs | Sort-Object DistinguishedName -Unique)
        If ($OUByName) {
            $NameMatches = $OUByName | Select Name, DistinguishedName
            Write-Host "Found organizational units by name match. Did you mean one of the following distinguished names?"
            Write-Table ($NameMatches)
            $DistinguishedNames = $OUByName | Select -ExpandProperty DistinguishedName
            $Context.Failures.Add("InferredOrganizationalUnit", $DistinguishedNames)
        } Else {
            Write-Host "No organizational units found by name match"
        }
    }

    $Context.Test++

    Return $Context
}
}

function Test-AdminGroup {
<#
.SYNOPSIS
    Test admin group works for FSx integration

.DESCRIPTION
    Test admin group works for FSx integration

.EXAMPLE
    $AdminGroup = 'FSxDelegatedAdmins'
    $AdDomain = [PSCustomObject]@{
        Name = 'test-ad'
        DNSRoot = 'test-ad.local'
    }
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
    $Context = @{
        AdminGroup = $AdminGroup
        AdDomain = $AdDomain
        Credential = $Credential
        Failures = @{}
        Skip = 0
        Test = 6
        TestHeaderColor = "Green"
    }
    Test-AdminGroup -Context $Context
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate Admin Group"

    If ($Context.Failures.Count -gt 0) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    If (!$Context.AdminGroup) {
        Write-Host "No admin group supplied, defaulting to 'Domain Admins'"
        $Context.AdminGroup = 'Domain Admins'
    }

    Try {
        Add-Type -AssemblyName System.DirectoryServices.AccountManagement
        $ContextType = [System.DirectoryServices.AccountManagement.ContextType]::Domain

        $PwdPointer = [Runtime.InteropServices.Marshal]::SecureStringToBSTR($Context.Credential.Password)
        $Password = [Runtime.InteropServices.Marshal]::PtrToStringAuto($PwdPointer)

        $PrincipalContext = New-Object System.DirectoryServices.AccountManagement.PrincipalContext($ContextType,
            $Context.AdDomain.DNSRoot,
            $Context.Credential.UserName,
            $Password)

        $IdentityType = [System.DirectoryServices.AccountManagement.IdentityType]::Name
        $GroupPrincipal = [System.DirectoryServices.AccountManagement.GroupPrincipal]::FindByIdentity($PrincipalContext,
            $IdentityType,
            $Context.AdminGroup)

        Write-Table ($GroupPrincipal | Select Name, DistinguishedName, SamAccountName, Members)

        if ($GroupPrincipal.DistinguishedName -like "*CN=Builtin*") {
            Write-Warning("Domain Admin group exists but it is in the Builtin container")
            $Context.Failures.Add("AdGroupInBuildinContainer",
                    $Context.AdminGroup)
        }

        $Context.Add("FSxAdminGroup", $GroupPrincipal)
    } Catch [System.Management.Automation.MethodInvocationException] {
        # Counter-intuitive error with message similar to below.
        # Exception calling "FindByIdentity" with "3" argument(s): "The specified directory service attribute or value does not exist."
        Write-Warning("Please validate service account user has 'Read' permission on both Users and Computers AD containers.")
        $Context.Failures.Add("UnauthorizedReadOnDefaultContainers", [PSCustomObject]@{
                ServiceAccount = $Context.Credential.UserName
                UsersContainer = $Context.AdDomain.UsersContainer
                ComputersContainer = $Context.AdDomain.ComputersContainer
            })
    } Catch {
        Write-Warning("Failed to retrieve domain admin group " + $Context.AdminGroup + " due to unknown exception. " + $_)
        $Context.Failures.Add("InvalidAdminGroup", $Context.AdminGroup)
    }

    $Context.Test++

    Return $Context
}
}

function Test-SubnetADSiteAssociation {
<#
.SYNOPSIS
    Test that provided EC2 Subnets belong to a single AD Site

.DESCRIPTION
    Test that provided EC2 Subnets belong to a single AD Site

.EXAMPLE
    $Subnet1 = [PSCustomObject]@{
                SubnetId = "subnet-0f30db1cad3a599d1"
                CidrBlock = "10.0.64.0/19"
            }

    $Subnet2 = [PSCustomObject]@{
                SubnetId = "subnet-04431191671ac0d19"
                CidrBlock = "10.0.0.0/19"
            }
    $ReachableDC = [PSCustomObject]@{
                Name = 'DC1.test-ad.local'
                IPAddress = '10.0.5.228'
            }
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
    $Context = @{
        Subnets = @($Subnet1, $Subnet2)
        ReachableDC = $ReachableDC
        Credential = $Credential
        Failures = @{}
        Warnings = @{}
        Skip = 0
        Test = 7
        TestHeaderColor = "Green"
    }
    Test-SubnetADSiteAssociation -Context $Context

.LINK
    Active Directory site topology
    https://docs.microsoft.com/en-us/windows-server/identity/ad-ds/plan/designing-the-site-topology
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate that provided EC2 Subnets belong to a single AD Site"

    If ($Context.Failures.Count -gt 0) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    $CommonAdReplicationArgs = @{
        Server = $Context.ReachableDC.IPAddress
        Credential = $Context.Credential
        ErrorAction = 'SilentlyContinue'
    }

    # First, determine the sites and subnets that correspond to the VPC Subnets associated with Amazon FSx
    $AdSubnets = (Get-ADReplicationSubnet -Filter * @CommonAdReplicationArgs | Sort-Object DistinguishedName -Unique)

    If ($AdSubnets) {
        Write-Table ($AdSubnets | Select Name, DistinguishedName, Site)

        $FSxAdSites = New-Object System.Collections.Generic.HashSet[string]
        $SubnetIdToAdSite = @{}
        $SubnetIdsWithNoSite = @()
        ForEach ($Ec2Subnet in $Context.Subnets) {
            $SubnetId = $Ec2Subnet.SubnetId

            # Find the AD site that each EC2 subnet corresponds to
            $MatchingAdSubnets = ($AdSubnets | Where-Object { Assert-CIDRContains -ParentCIDR $_.Name -ChildCIDR $Ec2Subnet.CidrBlock })

            If (!$MatchingAdSubnets) {
                Write-Warning "Subnet not defined in an Active Directory site: $SubnetId!"
                $SubnetIdsWithNoSite += $SubnetId
            } ElseIf ($MatchingAdSubnets.Count -gt 1) {
                # Select the most restrictive site (biggest bit mask length)
                $MatchingAdSubnets = ($MatchingAdSubnets | Sort-Object -Descending { [int](($_.Name -split "/")[1]) } | Select-Object -first 1)
            }

            If ($MatchingAdSubnets.Site) {
                $MatchedADSite = $MatchingAdSubnets.Site
                Write-Host "Best match for EC2 subnet $SubnetId is AD site $MatchedADSite"
                $SubnetIdToAdSite[$SubnetId] = $MatchedADSite
                $FSxAdSites.Add($MatchedADSite)
            }
        }

        If ($SubnetIdsWithNoSite.Count -gt 0) {
            Write-Warning "The following subnet(s) are not defined in an Active Directory site: $SubnetIdsWithNoSite! Please ensure all subnets in the VPC associated with your Amazon FSx file system are defined in an Active Directory site."
            $Context.Failures.Add("NoAdSubnetForEc2Subnet", $SubnetIdsWithNoSite)
            $Context.Test++
            return $Context
        }

        If ($Context.Failures.Count -eq 0 -and ($FSxAdSites.Count -gt 1)) {
            $SubnetIds = ($Context.Subnets | Select -ExpandProperty SubnetId)
            Write-Warning "EC2 subnets $SubnetIds matched to different AD sites! Make sure they are in a single AD site."
            $Context.Failures.Add("SubnetsInSeparateAdSites", $SubnetIdToAdSite)
        }
    } Else {
        # If there are no subnets, assume there is only one site in this AD
        Write-Host "`nNo AD Subnets found in the Active Directory. FSx will look for the default AD Site.`n"
        $FSxAdSites = (Get-ADReplicationSite -Filter * @CommonAdReplicationArgs | Select -ExpandProperty DistinguishedName)
        If (!$FSxAdSites) {
            Write-Warning "Default AD site not found!"
            $Context.Failures.Add("NoAdSites", 1)
        } ElseIf ($FSxAdSites.Count -gt 1) {
            Write-Warning ("Found " + $FSxAdSites.Count + " AD sites without subnets. Make sure each subnet is mapped to a single site")
            $Context.Failures.Add("MultipleAdSitesWithoutSubnets", $FSxAdSites)
        }
    }

    If ($FSxAdSites -and $FSxAdSites.Count -gt 0) {
        $FSxAdSite = @($FSxAdSites)[0]
        $FSxAdSiteName = $FSxAdSite.Split(",")[0].Split("=")[1]
        Write-Host "The EC2 subnets provided correspond to the following AD site: ${FSxAdSiteName}`n"
        $Context.Add("FSxAdSite", $FSxAdSite)
        $Context.Add("FSxAdSiteName", $FSxAdSiteName)
    }

    $Context.Test++

    Return $Context
}
}

function Test-CreateComputer {
<#
.SYNOPSIS
    Test that FSx service account has Create Computer Objects permission

.DESCRIPTION
    Test that FSx service account has Create Computer Objects permission
    on either the supplied OU or the Computers Container for the domain

.EXAMPLE
    $OrganizationalUnit = 'OU=Amazon FSx,OU=test-ad,DC=test-ad,DC=local'
    $SIDValue = 'S-1-5-21-1224719988-2079912999-2002669104'
    $DomainSID = [PSCustomObject]@{
        BinaryLength = 24
        AccountDomainSid = $SIDValue
        Value = $SIDValue
    }
    $AdDomain = [PSCustomObject]@{
        Name = 'test-ad'
        DNSRoot = 'test-ad.local'
        DomainSID = $DomainSID
    }
    $DC1 = [PSCustomObject]@{
            Name = 'DC1.test-ad.local'
            IPAddress = '10.0.5.228'
        }
    $DC2 = [PSCustomObject]@{
            Name = 'DC2.test-ad.local'
            IPAddress = '10.0.77.202'
        }
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
    $FSxAdSiteName = 'Site1'
    $Context = @{
        OrganizationalUnit = $OrganizationalUnit
        AdDomain = $AdDomain
        DomainControllers = @($DC1, $DC2)
        Credential = $Credential
        FSxAdSiteName = $FSxAdSiteName
        Failures = @{}
        Warnings = @{}
        Skip = 0
        Test = 10
        TestHeaderColor = "Green"
    }
    Test-CreateComputer -Context $Context

.LINK
    Delegating Privileges to Your Amazon FSx Service Account
    https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-managed-AD-best-practices.html#connect_delegate_privileges
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate 'Create Computer Objects' permission"

    If (!$Context.TestServiceAccountPermissions) {
        Write-Host "Testing service account permissions is disabled, skipping all service account tests..." -ForegroundColor $Context.TestHeaderColor
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    If ($Context.Failures.Count -gt 0) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    $MaxAttemptsPerDC = 10
    $TestComputerDescription = "[Amazon FSx] Test computer object created to test AD permissions"
    $TestComputerNamePrefix = "amznfsxtest"

    $CommonCreateArgs = @{
        Path = $Context.OrganizationalUnit
        Description = $TestComputerDescription
        Enabled = $True
        Credential = $Context.Credential
        PassThru = $True
        ErrorAction = 'SilentlyContinue'
    }

    $ReadOnlyDC = "ReadOnlyDC"
    $CreateComputer = "UnauthorizedCreateComputerObjectsOnOU"
    # Try all DCs in case some are read-only
    # Note: these are already scoped down to the mapped AD site and therefore
    # should be reasonably bounded in terms of time
    ForEach ($DC in $Context.DomainControllers) {
        For ($Attempt = 0; $Attempt -lt $MaxAttemptsPerDC; $Attempt++) {
            Try {
                $FoundUnusedComputerName = $True
                $ComputerName = ($TestComputerNamePrefix + ('{0:x4}' -f (Get-Random -Maximum 65535)))
                $TestComputerObject = (New-ADComputer -Name $ComputerName -Server $DC.IPAddress @CommonCreateArgs)
                $Context.WriteableSiteDC = $DC
            }
            Catch [Microsoft.ActiveDirectory.Management.ADReferralException] {
                # A referral was returned from the server
                Write-Warning ("Unable to create a computer object because the following AD Domain Controller is read-only: " + $DC.Name + " (" + $DC.IPAddress + ")")
                $FailureReason = $ReadOnlyDC
            }
            Catch [Microsoft.ActiveDirectory.Management.ADIdentityAlreadyExistsException] {
                # The specified account already exists
                $FoundUnusedComputerName = $False
            }
            Catch {
                $LastError = $_
                # All of the errors below indicate that the AD is not able to process the request
                Write-Warning ("Unable to create a computer object using the following AD Domain Controller: " + $DC.Name + " (" + $DC.IPAddress + ")")
                If ($LastError.Exception -match "The server is unwilling to process the request") {
                    Write-Warning("Please make sure service account has ListChildren permissions on " + $Context.OrganizationalUnit)
                    $FailureReason = "UnauthorizedListChildrenOnOU"
                } Else {
                    $FailureReason = $CreateComputer
                }
            }
            If ($FoundUnusedComputerName) {
                # All cases other than the ADIdentityAlreadyExistsException won't change with
                # a retry on the same DC, move onto the next DC
                Break
            }
            Sleep 1
        }

        If (!$FoundUnusedComputerName) {
            Write-Warning ("Unable to find available test computer name with ${MaxAttemptsPerDC} attempts in " + $Context.OrganizationalUnit)
            $Context.Failures.Add("NoAvailableTestComputerName", $Context.OrganizationalUnit)
            Break
        }

        If ($TestComputerObject) {
            Break
        }
    }

    If ($TestComputerObject) {
        Write-Table ($TestComputerObject | Select Name, DistinguishedName, SID)

        $Context.TestComputerObject = $TestComputerObject
    } ElseIf ($FoundUnusedComputerName) {
        If ($FailureReason -eq $ReadOnlyDC) {
            Write-Warning ("All domain controllers are read-only in site " + $Context.FSxAdSiteName)
            $Context.Failures.Add("ReadOnlyDCsInSite", $Context.DomainControllers)
        } Else {
            $Context.Failures.Add($FailureReason, $Context.OrganizationalUnit)
            If ($FailureReason -eq $CreateComputer) {
                Write-Warning ("Please verify service account has 'Create Computer Objects' permission on " + $Context.OrganizationalUnit)
            }
        }
    }

    $Context.Test++

    Return $Context
}
}

function Test-WriteDnsHostName {
<#
.SYNOPSIS
    Test that FSx service account has permission for 'Validated write to DNS host name'

.DESCRIPTION
    Test that FSx service account has permission for 'Validated write to DNS host name'
    on either the supplied OU or the Computers Container for the domain

.EXAMPLE
    $TestComputerName = 'amznfsxtest7153'
    $OrganizationalUnit = 'OU=Amazon FSx,OU=test-ad,DC=test-ad,DC=local'
    $AdDomain = [PSCustomObject]@{
        Name = 'test-ad'
        DNSRoot = 'test-ad.local'
        DistinguishedName = 'DC=test-ad,DC=local'
        ComputersContainer = 'CN=Computers,DC=test-ad,DC=local'
    }
    $TestComputerObject = [PSCustomObject]@{
            Name = $TestComputerName
            DistinguishedName = "CN=$TestComputerName,$OrganizationalUnit"
        }
    $WriteableSiteDC = [PSCustomObject]@{
            Name = 'DC1.test-ad.local'
            IPAddress = '10.0.5.228'
        }
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
    $Context = @{
        OrganizationalUnit = $OrganizationalUnit
        AdDomain = $AdDomain
        TestComputerObject = $TestComputerObject
        WriteableSiteDC = $WriteableSiteDC
        Credential = $Credential
        Failures = @{}
        Warnings = @{}
        Skip = 0
        Test = 11
        TestHeaderColor = "Green"
    }
    Test-WriteDnsHostName -Context $Context

.LINK
    Delegating Privileges to Your Amazon FSx Service Account
    https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-managed-AD-best-practices.html#connect_delegate_privileges
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate 'Validated write to DNS host name' permission"
    If (!$Context.TestComputerObject) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    $DnsHostName = ($Context.TestComputerObject.Name + "." + $Context.AdDomain.DNSRoot)

    $SetAdArgs = @{
        Identity = $Context.TestComputerObject.DistinguishedName
        DNSHostName = $DnsHostName
        Server = $Context.WriteableSiteDC.IPAddress
        Credential = $Context.Credential
        PassThru = $True
        ErrorAction = 'SilentlyContinue'
    }

    Try {
        $Updated = (Set-ADComputer @SetAdArgs)
        Write-Table ($Updated | Select Name, DNSHostName)
    }
    Catch {
        Write-Warning ("Please verify service account has 'Validated write to DNS host name' permission on " + $Context.OrganizationalUnit)
        $Context.Failures.Add("UnauthorizedWriteDnsHostNameOnOU", $Context.OrganizationalUnit)
    }

    $Context.Test++

    Return $Context
}
}

function Test-WriteServicePrincipalName {
<#
.SYNOPSIS
    Test that FSx service account has permission for 'Validated write to service principal name'

.DESCRIPTION
    Test that FSx service account has permission for 'Validated write to service principal name'
    on either the supplied OU or the Computers Container for the domain

.EXAMPLE
    $TestComputerName = 'amznfsxtest7153'
    $OrganizationalUnit = 'OU=Amazon FSx,OU=test-ad,DC=test-ad,DC=local'
    $TestComputerObject = [PSCustomObject]@{
            Name = $TestComputerName
            DistinguishedName = "CN=$TestComputerName,$OrganizationalUnit"
        }
    $WriteableSiteDC = [PSCustomObject]@{
            Name = 'DC1.test-ad.local'
            IPAddress = '10.0.5.228'
        }
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
    $Context = @{
        OrganizationalUnit = $OrganizationalUnit
        TestComputerObject = $TestComputerObject
        WriteableSiteDC = $WriteableSiteDC
        Credential = $Credential
        Failures = @{}
        Warnings = @{}
        Skip = 0
        Test = 12
        TestHeaderColor = "Green"
    }
    Test-WriteServicePrincipalName -Context $Context

.LINK
    Delegating Privileges to Your Amazon FSx Service Account
    https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-managed-AD-best-practices.html#connect_delegate_privileges
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate 'Validated write to service principal name' permission"
    If (!$Context.TestComputerObject) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }
    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    $ServicePrincipalNames = @{
        Add = ("HOST/" + $Context.TestComputerObject.Name)
    }

    $CommonAdArgs = @{
        Identity = $Context.TestComputerObject.DistinguishedName
        Server = $Context.WriteableSiteDC.IPAddress
        Credential = $Context.Credential
        ErrorAction = 'SilentlyContinue'
    }

    Try {
        Set-ADComputer @CommonAdArgs -ServicePrincipalNames $ServicePrincipalNames
        $Updated = Get-ADComputer @CommonAdArgs -Properties ServicePrincipalNames
        Write-Table ($Updated | Select Name, ServicePrincipalNames)
    }
    Catch {
        Write-Warning ("Please verify service account has 'Validated write to service principal name' permission on " + $Context.OrganizationalUnit)
        $Context.Failures.Add("UnauthorizedWriteServicePrincipalNameOnOU", $Context.OrganizationalUnit)
    }

    $Context.Test++

    Return $Context
}
}

function Test-ResetPassword {
<#
.SYNOPSIS
    Test that FSx service account has permission for 'Reset Password'

.DESCRIPTION
    Test that FSx service account has permission for 'Reset Password'
    on either the supplied OU or the Computers Container for the domain

.EXAMPLE
    $TestComputerName = 'amznfsxtest7153'
    $OrganizationalUnit = 'OU=Amazon FSx,OU=test-ad,DC=test-ad,DC=local'
    $TestComputerObject = [PSCustomObject]@{
            Name = $TestComputerName
            DistinguishedName = "CN=$TestComputerName,$OrganizationalUnit"
        }
    $WriteableSiteDC = [PSCustomObject]@{
            Name = 'DC1.test-ad.local'
            IPAddress = '10.0.5.228'
        }
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
    $Context = @{
        OrganizationalUnit = $OrganizationalUnit
        TestComputerObject = $TestComputerObject
        WriteableSiteDC = $WriteableSiteDC
        Credential = $Credential
        Failures = @{}
        Warnings = @{}
        Skip = 0
        Test = 13
        TestHeaderColor = "Green"
    }
    Test-ResetPassword -Context $Context

.LINK
    Delegating Privileges to Your Amazon FSx Service Account
    https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-managed-AD-best-practices.html#connect_delegate_privileges
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate 'Reset Password' permission"
    If (!$Context.TestComputerObject) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    $PasswordLength = [int32](Get-Random -Minimum 20 -Maximum 32)
    $NumNonAlphaNumericChars = [int32]($PasswordLength / 2)

    Add-Type -AssemblyName 'System.Web'
    $NewPassword = [System.Web.Security.Membership]::GeneratePassword($PasswordLength, $NumNonAlphaNumericChars)
    $SecureNewPassword = (ConvertTo-SecureString -AsPlainText $NewPassword -Force)

    $CommonAdArgs = @{
        Identity = $Context.TestComputerObject.DistinguishedName
        Server = $Context.WriteableSiteDC.IPAddress
        Credential = $Context.Credential
        ErrorAction = 'SilentlyContinue'
    }

    Try {
        Set-ADAccountPassword -Reset -NewPassword $SecureNewPassword @CommonAdArgs
        $Context.TestComputerPassword = $SecureNewPassword
        $Updated = Get-ADComputer @CommonAdArgs -Properties PasswordLastSet
        Write-Table ($Updated | Select Name, PasswordLastSet)
    }
    Catch {
        Write-Warning ("Please verify service account has 'Reset Password' permission on " + $Context.OrganizationalUnit)
        $Context.Failures.Add("UnauthorizedResetPasswordOnOU", $Context.OrganizationalUnit)
    }

    $Context.Test++

    Return $Context
}
}

function Test-ListOrganization {
<#
.SYNOPSIS
    Test that 'This Organization' has ListChildren permission on organizational unit

.DESCRIPTION
    Test that 'This Organization' has ListChildren permission
    on either the supplied OU or the Computers Container for the domain

.EXAMPLE
    $TestComputerName = 'amznfsxtest7153'
    $OrganizationalUnit = 'OU=Amazon FSx,OU=test-ad,DC=test-ad,DC=local'
    $TestComputerObject = [PSCustomObject]@{
            Name = $TestComputerName
            DistinguishedName = "CN=$TestComputerName,$OrganizationalUnit"
        }
    $WriteableSiteDC = [PSCustomObject]@{
            Name = 'DC1.test-ad.local'
            IPAddress = '10.0.5.228'
        }
    $ComputerObjectPassword = ConvertTo-SecureString 'ComputerObjectPassword' -AsPlainText -Force
    $Context = @{
        OrganizationalUnit = $OrganizationalUnit
        TestComputerObject = $TestComputerObject
        WriteableSiteDC = $WriteableSiteDC
        Credential = $Credential
        Failures = @{}
        Warnings = @{}
        Skip = 0
        Test = 14
        TestHeaderColor = "Green"
    }
    Test-ListOrganization -Context $Context

.LINK
    TODO needs doc update!
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate 'This Organization' list children permission"
    If (!$Context.TestComputerPassword) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    $ComputerObjectCredential = New-Object System.Management.Automation.PSCredential ($Context.TestComputerObject.SamAccountName, $Context.TestComputerPassword)

    $CommonAdArgs = @{
        Identity = $Context.TestComputerObject.DistinguishedName
        Server = $Context.WriteableSiteDC.IPAddress
        Credential = $ComputerObjectCredential
        ErrorAction = 'SilentlyContinue'
    }

    Try {
        $Computer = Get-ADComputer @CommonAdArgs
        Write-Table ($Computer | Select Name, DistinguishedName)
    }
    Catch {
        Write-Warning ("Please verify 'This Organization' has ListChildren on " + $Context.OrganizationalUnit)
        $Context.Failures.Add("UnauthorizedThisOrganizationListChildrenOnOU", $Context.OrganizationalUnit)
    }

    $Context.Test++

    Return $Context
}
}

function Test-ReadWriteAccountRestrictions {
<#
.SYNOPSIS
    Test that FSx service account has permission for 'Read and write Account Restrictions'

.DESCRIPTION
    Test that FSx service account has permission for 'Read and write Account Restrictions'
    on either the supplied OU or the Computers Container for the domain

.EXAMPLE
    $TestComputerName = 'amznfsxtest7153'
    $OrganizationalUnit = 'OU=Amazon FSx,OU=test-ad,DC=test-ad,DC=local'
    $TestComputerObject = [PSCustomObject]@{
            Name = $TestComputerName
            DistinguishedName = "CN=$TestComputerName,$OrganizationalUnit"
        }
    $WriteableSiteDC = [PSCustomObject]@{
            Name = 'DC1.test-ad.local'
            IPAddress = '10.0.5.228'
        }
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
    $Context = @{
        OrganizationalUnit = $OrganizationalUnit
        TestComputerObject = $TestComputerObject
        WriteableSiteDC = $WriteableSiteDC
        Credential = $Credential
        Failures = @{}
        Warnings = @{}
        Skip = 0
        Test = 15
        TestHeaderColor = "Green"
    }
    Test-ReadWriteAccountRestrictions -Context $Context

.LINK
    Delegating Privileges to Your Amazon FSx Service Account
    https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-managed-AD-best-practices.html#connect_delegate_privileges
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate 'Read and write Account Restrictions' permission"
    If (!$Context.TestComputerObject) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    $CommonAdArgs = @{
        Identity = $Context.TestComputerObject.DistinguishedName
        Server = $Context.WriteableSiteDC.IPAddress
        Credential = $Context.Credential
        ErrorAction = 'SilentlyContinue'
    }

    Try {
        Set-ADAccountControl @CommonAdArgs -PasswordNotRequired $False
        $Updated = Get-ADComputer @CommonAdArgs -Properties PasswordNotRequired
        Write-Table ($Updated | Select Name, PasswordNotRequired)
    }
    Catch {
        Write-Warning ("Please verify service account has 'Read and write Account Restrictions' permission on " + $Context.OrganizationalUnit)
        $Context.Failures.Add("UnauthorizedRestrictAccountsOnOU", $Context.OrganizationalUnit)
    }

    $Context.Test++

    Return $Context
}
}
function Test-ModifyADComputerPermission {
    <#
 .SYNOPSIS
  Test that FSx service account has 'WriteDacl' permission.
 .DESCRIPTION
  Test that FSx service account has 'WriteDacl' permission.
 .EXAMPLE
  $AdUser = [PSCustomObject]@{
   Name = 'fsxServiceUser'
   DistinguishedName = "CN=fsxServiceUser,DC=test-ad,DC=local"
  }
  $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
  $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
  $Context = @{
   AdUser = $AdUser
   Credential = $Credential
   Failures = @{}
   Warnings = @{}
   Skip = 0
   Test = 17
   TestHeaderColor = "Green"
  }
  Test-ModifyPermission -Context $Context
 .LINK
  Delegating Privileges to Your Amazon FSx Service Account
  https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-managed-AD-best-practices.html#connect_delegate_privileges
 #>
    [CmdletBinding()]
    [OutputType([object])]
    param (
        [Parameter(Mandatory = $true, Position = 0)]
        [object]$Context
    )
    $TestName = "Validate 'Modify computer ACL' permission"
    If (!$Context.AdUser) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }
    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor
    try {
        #Getting the service account name from $credential variable
        $ServiceAccount = $Credential.UserName
        $ComputerDN = $Context.TestComputerObject.DistinguishedName
        $ComputerName = $Context.TestComputerObject.Name ; $Credential = $context.Credential
        # Get Domain name from DomainDNSRoot
        $Words = $DomainDNSRoot -split '\.'
        $DomainName = $Words[0].toUpper()
        # Get the current ACL
        $acl = Get-Acl -Path "AD:$ComputerDN"
        #Remove Read permission if exits to test the service account modify permission
        if ($Acl.Access | Where-Object { $_.IdentityReference -eq $DomainName + '\' + $ServiceAccount -and $_.ActiveDirectoryRights -like '*ReadProperty*' }) {
            $sid = Get-ADUser -Identity $ServiceAccount | select -ExpandProperty SID
            $ctrlType = [System.Security.AccessControl.AccessControlType]::Allow
            $rights = [System.DirectoryServices.ActiveDirectoryRights]::ReadProperty
            $rule = New-Object System.DirectoryServices.ActiveDirectoryAccessRule($sid, $rights, $ctrlType)
            $Acl.removeaccessrule($rule)
            Write-Host "Modifying Computer object Permission using Set-Acl"
            Set-Acl -Path AD:\$ComputerDN -AclObject $Acl
        }
        #Display Service Account Permission on Computer Object Before Modifications
        if ($Acl.Access | Where-Object { $_.IdentityReference -eq $DomainName + '\' + $ServiceAccount }) {
            Write-Host "Service Account Permission on Computer Object Before Modifications"
            Write-Table($Acl.Access | Where-Object { $_.IdentityReference -eq $DomainName + '\' + $ServiceAccount } )
        }
        # $script adds Read control to the service account on computer object
        # Running the script with service account credentials
        $job = Invoke-Command -ComputerName localhost -Credential $credential -ArgumentList $ComputerDN , $ServiceAccount -ScriptBlock {
            param($ComputerDN , $ServiceAccount)
            Import-Module ActiveDirectory
            Sleep 5
            $acl = Get-acl -Path AD:\$ComputerDN
            $sid = Get-ADUser -Identity $ServiceAccount | select -ExpandProperty SID
            $ctrlType = [System.Security.AccessControl.AccessControlType]::Allow
            $rights = [System.DirectoryServices.ActiveDirectoryRights]::ReadProperty
            $rule = New-Object System.DirectoryServices.ActiveDirectoryAccessRule($sid, $rights, $ctrlType)
            $acl.Addaccessrule($rule)
            Write-Host "Modifying Computer object Permission using Set-Acl"
            Set-Acl -Path AD:\$ComputerDN -AclObject $acl
        } -AsJob
        Wait-Job -job $Job
        $modifiedAcl = Get-Acl -Path "AD:\$ComputerDN"
        if ($modifiedAcl.Access | Where-Object { $_.IdentityReference -eq $DomainName + '\' + $ServiceAccount -and $_.ActiveDirectoryRights -like '*ReadProperty*' }) {
            # Check if permission was added successfully
            Write-Host "Service Account Permission on Computer Object After Modifications"
            Write-Table($modifiedAcl.Access | Where-Object { $_.IdentityReference -eq $DomainName + '\' + $ServiceAccount } )
            Write-Host "Service account '$ServiceAccount' has 'Modify computer ACL' permission on computer object." -ForegroundColor Green
        }
        else {
            Write-Host "Permission modification failed."
            Write-Host "Service account '$ServiceAccount' does not have 'Modify computer ACL' permission." -ForegroundColor Red
        }
    }
    catch {
        Write-Warning $_.Exception.Message
        $Context.Failures.Add("UnauthorizedWriteDacl", $Context.AdUser.DistinguishedName)
    }
    $Context.Test++
    return $Context
}

function Test-DeleteComputer {
<#
.SYNOPSIS
    Test that FSx service account has permission for 'Delete Computer Objects'

.DESCRIPTION
    Test that FSx service account has permission for 'Delete Computer Objects'
    on either the supplied OU or the Computers Container for the domain

.EXAMPLE
    $TestComputerName = 'amznfsxtest7153'
    $OrganizationalUnit = 'OU=Amazon FSx,OU=test-ad,DC=test-ad,DC=local'
    $TestComputerObject = [PSCustomObject]@{
            Name = $TestComputerName
            DistinguishedName = "CN=$TestComputerName,$OrganizationalUnit"
        }
    $WriteableSiteDC = [PSCustomObject]@{
            Name = 'DC1.test-ad.local'
            IPAddress = '10.0.5.228'
        }
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
    $Context = @{
        OrganizationalUnit = $OrganizationalUnit
        TestComputerObject = $TestComputerObject
        WriteableSiteDC = $WriteableSiteDC
        Credential = $Credential
        Failures = @{}
        Warnings = @{}
        Skip = 0
        Test = 16
        TestHeaderColor = "Green"
    }
    Test-DeleteComputer -Context $Context

.LINK
    Delegating Privileges to Your Amazon FSx Service Account
    https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-managed-AD-best-practices.html#connect_delegate_privileges
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate 'Delete Computer Objects' permission"
    If (!$Context.TestComputerObject) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    $CommonAdArgs = @{
        Identity = $Context.TestComputerObject.DistinguishedName
        Server = $Context.WriteableSiteDC.IPAddress
        Credential = $Context.Credential
        ErrorAction = 'SilentlyContinue'
    }

    $ComputerName = $Context.TestComputerObject.Name
    Try {
        Remove-ADComputer @CommonAdArgs -Confirm:$False
        Write-Host "`nTest computer object $ComputerName deleted!`n"
    }
    Catch [Microsoft.ActiveDirectory.Management.ADIdentityNotFoundException] {
        # Directory object not found
        Write-Warning "Test computer object $ComputerName is already deleted! Considering delete test skipped."
        $Context.Skip++
        Return $Context
    }
    Catch {
        Write-Warning ("Please verify service account has 'Delete Computer Objects' permission on " + $Context.OrganizationalUnit)
        Write-Warning ("Please manually delete test computer object " + $Context.TestComputerObject.DistinguishedName)
        $Context.Failures.UnauthorizedDeleteComputerOnOU = [PSCustomObject]@{
            OrganizationalUnit = $Context.OrganizationalUnit
            Computer = $Context.TestComputerObject
        }
    }

    $Context.Test++

    Return $Context
}
}
#Requires -Modules ActiveDirectory


function Test-FSxADConfiguration {
<#
.SYNOPSIS
    Test Active Directory configuration for ability to integrate with Amazon FSx

.DESCRIPTION
    Test Active Directory configuration for ability to integrate with Amazon FSx
    Automates the pre-requisite checks as listed in Amazon FSx documentation

    Please resolve all failures prior to creating an Amazon FSx
    file system. It is also HIGHLY recommended to resolve all warnings
    which may cause degradation of service.

.EXAMPLE
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ('fsxServiceUser', $Password)
    $DomainDNSRoot = 'test-ad.local'
    $DnsIpAddresses = @('10.0.88.17', '10.0.77.202')
    $SubnetIds = @('subnet-04431191671ac0d19', 'subnet-0f30db1cad3a599d1')

    $Result = Test-FSxADConfiguration -Credential $Credential -DomainDNSRoot $DomainDNSRoot -DnsIpAddresses $DnsIpAddresses -SubnetIds $SubnetIds

.EXAMPLE
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ('fsxServiceUser', $Password)
    $FSxADValidationArgs = @{
        DomainDNSRoot = 'test-ad.local'
        DnsIpAddresses = @('10.0.88.17', '10.0.77.202')
        SubnetIds = @('subnet-04431191671ac0d19', 'subnet-0f30db1cad3a599d1')

        Credential = $Credential
    }
    # Optional arguments
    $FSxADValidationArgs['OrganizationalUnit'] = 'OU=Amazon FSx,OU=test-ad,DC=test-ad,DC=local'
    $FSxADValidationArgs['AdminGroup'] = 'FSxDelegatedAdmins'
    $FSxADValidationArgs['TestServiceAccountPermissions'] = $True
    $FSxADValidationArgs['TranscriptDirectory'] = [Environment]::GetFolderPath("MyDocuments")
    $FSxADValidationArgs['DomainControllersMaxCount'] = 100
    $FSxADValidationArgs['ComputerObjects'] = @('amznfsx123', 'amznfsx456')

    $Result = Test-FSxADConfiguration @FSxADValidationArgs

.LINK
    Working with Active Directory in Amazon FSx for Windows File Server
    https://docs.aws.amazon.com/fsx/latest/WindowsGuide/aws-ad-integration-fsxW.html

    Prerequisites for Using a Self-Managed Microsoft AD
    https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-manage-prereqs.html
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [ValidateNotNullOrEmpty()]
    [string]
    $DomainDNSRoot, # DNS root of Active Directory domain

    [Parameter(Mandatory = $True,
               Position = 1)]
    [ValidatePattern("^(([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])\.){3}([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])$")]
    [string[]]
    $DnsIpAddresses, # IP v4 addresses of DNS servers

    [Parameter(Mandatory = $True,
               Position = 2)]
    [ValidatePattern("^(subnet-[0-9a-f]{8,})$")]
    [string[]]
    $SubnetIds, # Subnet IDs for Amazon FSx file server(s)

    [Parameter(Mandatory = $True,
               Position = 3)]
    [System.Management.Automation.PSCredential]
    $Credential, # Credential for Amazon FSx service account

    [Parameter(Mandatory = $False,
               Position = 4)]
    [string]
    $OrganizationalUnit, # Distinguished name for Organizational Unit (OU) within the domain that you want your Amazon FSx file system to join to. Defaults to Computers container

    [Parameter(Mandatory = $False,
               Position = 5)]
    [string]
    $AdminGroup, # The name of the domain group whose members are granted administrative privileges for the Amazon FSx file system. Defaults to "Domain Admins"

    [Parameter(Mandatory = $False,
               Position = 6)]
    [Switch]
    $TestServiceAccountPermissions, # Testing service account permissions include creating computer objects in your AD, opt-in

    [Parameter(Mandatory = $False,
               Position = 7)]
    [string]
    $TranscriptDirectory, # Directory to log output to. If not specified, no transcript will be generated

    [Parameter(Mandatory = $False,
               Position = 8)]
    [ValidateRange(1, 10000)]
    [int]
    $DomainControllersMaxCount, # The maximum amount of domain controllers for which connectivity will be tested. If not specified, the default value is set to 100

    [Parameter(Mandatory = $False,
            Position = 9)]
    [String[]]
    $ComputerObjects
)
Process {
    $Context = @{
        Domain = $DomainDNSRoot
        DnsIpAddresses = $DnsIpAddresses
        SubnetIds = $SubnetIds
        Credential = $Credential

        OrganizationalUnit = $OrganizationalUnit
        AdminGroup = $AdminGroup

        ComputerObjects = $ComputerObjects

        Failures = @{}
        Warnings = @{}
        Skip = 0
        Test = 1
        TestHeaderColor = "Green"
    }
    If ($TestServiceAccountPermissions) {
        $Context.TestServiceAccountPermissions = $True
    } Else {
        $ServiceAccountPrompt = "Testing service account permissions is not currently enabled. " +
            "If enabled, script will create test Active Directory computer objects in the organizational unit. " +
            "This will be cleaned up by the script unless delete permissions are " +
            "not properly configured on the provided service account " +
            "in which case manual cleanup may be necessary. Do you want to enable testing? [y/n]"
        $TestServiceAccount = Read-Host $ServiceAccountPrompt
        $Context.TestServiceAccountPermissions = ($TestServiceAccount -eq 'y')
    }

    If ($TranscriptDirectory) {
        $StartTime = (Get-Date -UFormat %Y-%m-%d-%H-%M-%S)
        $TranscriptPath = "${TranscriptDirectory}\Test-FSxADConfiguration-${StartTime}.txt"
        Write-Host "Recording transcript to ${TranscriptPath}"
        Start-Transcript -Path $TranscriptPath -IncludeInvocationHeader
    }

    If ($DomainControllersMaxCount) {
        $Context.DomainControllersMaxCount = $DomainControllersMaxCount
    } Else {
        $Context.DomainControllersMaxCount = 100
    }

    Write-Host "Running Active Directory validation with following input parameters: "
    [PSCustomObject]@{
        DomainDNSRoot = $DomainDNSRoot
        DnsIpAddresses = $DnsIpAddresses
        SubnetIds = $SubnetIds
        OrganizationalUnit = $OrganizationalUnit
        AdminGroup = $AdminGroup
        ComputerObjects = $ComputerObjects
        TestServiceAccountPermissions = $TestServiceAccountPermissions
    } | Format-List | Out-String | Write-Host

    # Basic validation

    # 1
    # Inputs: SubnetIds
    # Outputs: Subnets, VPC
    $Context = (Test-Subnets -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 2
    # Inputs: DnsIpAddresses, Domain, VPC
    # Outputs: DomainControllers, Forest
    $Context = (Test-Dns -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 3
    # Inputs: Credential, Domain, DomainControllers, VPC
    # Outputs: AdDomain, AdUser, ReachableDC
    $Context = (Test-Credential -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 4
    # Inputs: DomainControllers, DomainControllersMaxCount
    # Outputs: ControllersPingTestReport
    $Context = (Test-PingControllers -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 5
    # Inputs: AdDomain
    # Outputs: N/A
    $Context = (Test-Domain -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 6
    # Inputs: AdDomain, Credential, OrganizationalUnit, ReachableDC
    # Outputs: OU or DefaultComputers
    $Context = (Test-OrganizationalUnit -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 7
    # Inputs: AdDomain, AdminGroup, Credential
    # Outputs: FSxAdminGroup
    $Context = (Test-AdminGroup -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # AD site validation

    # 8
    # Inputs: Credential, ReachableDC, Subnets
    # Outputs: FSxAdSite, FSxAdSiteName
    $Context = (Test-SubnetADSiteAssociation -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 9 resolve DNS again scoped to AD site
    # Inputs: DnsIpAddresses, Domain, FSxAdSiteName, VPC
    # Outputs: DomainControllers
    $Context = (Test-Dns -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 10 run domain controller reachability again scoped to AD site
    # Inputs: Credential, Domain, DomainControllers, FSxAdSiteName, VPC
    # Outputs: ReachableSiteDC
    $Context = (Test-Credential -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # Service account validation (requires TestServiceAccountPermissions opt-in)

    # 11
    # Inputs: AdDomain, Credential, DomainControllers, FSxAdSiteName, OrganizationalUnit, TestServiceAccountPermissions
    # Outputs: TestComputerObject, WriteableSiteDC
    $Context = (Test-CreateComputer -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 12
    # Inputs: AdDomain, Credential, OrganizationalUnit, TestComputerObject, WriteableSiteDC
    # Outputs: N/A
    $Context = (Test-WriteDnsHostName -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 13
    # Inputs: Credential, OrganizationalUnit, TestComputerObject, WriteableSiteDC
    # Outputs: N/A
    $Context = (Test-WriteServicePrincipalName -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 14
    # Inputs: Credential, OrganizationalUnit, TestComputerObject, WriteableSiteDC
    # Outputs: TestComputerPassword
    $Context = (Test-ResetPassword -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 15
    # Inputs: OrganizationalUnit, TestComputerObject, TestComputerPassword, WriteableSiteDC
    # Outputs: N/A
    $Context = (Test-ListOrganization -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 16
    # Inputs: Credential, OrganizationalUnit, TestComputerObject, WriteableSiteDC
    # Outputs: N/A
    $Context = (Test-ReadWriteAccountRestrictions -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 17
    # Inputs: Credential, OrganizationalUnit, TestComputerObject, WriteableSiteDC
    # Outputs: N/A
    $Context = (Test-ModifyADComputerPermission -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 18
    # Inputs: Credential, OrganizationalUnit, TestComputerObject, WriteableSiteDC
    # Outputs: N/A
    $Context = (Test-DeleteComputer -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # Computer Object validation

    # 19
    # Inputs: AdDomain, ComputerObjects, Credential, OrganizationalUnit, ReachableDC
    # Outputs: N/A
    $Context = (Test-ComputerObjects -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    # 20
    # Inputs: DnsIpAddresses, VPC, Forest
    # Outputs: N/A
    $Context = (Test-GlobalCatalog -Context $Context | Where-Object { $_ -is [hashtable] } | Select-Object -first 1)

    $NumTests = 20
    $Skip = $Context.Skip

    If ($Skip -gt 0) {
        Write-Host "$Skip of $NumTests tests skipped." -ForegroundColor $Context.TestHeaderColor
    }

    $HasWarnings = ($Context.Warnings.Count -gt 0)
    If ($HasWarnings) {
        Write-Host "Possible configuration issues identified. Please see warning details below:" -ForegroundColor $Context.TestHeaderColor

        Write-Table ($Context.Warnings)
    }

    $DomainControllersUnreachable = ($Context.PingControllersResult.FailedDomainControllers.Count -gt 0)
    If ($DomainControllersUnreachable) {
        $Context.Failures.Add("UnreachableDomainControllers", $Context.PingControllersResult.FailedDomainControllers)
    }

    $HasNoErrors = ($Context.Failures.Count -eq 0)
    If ($HasNoErrors) {
        If ($HasWarnings) {
            $SuccessMessage = "SUCCESS with WARNINGS - All tests passed but there were possible configuration issues. It is HIGHLY recommended to address these before creating an Amazon FSx file system."
        } Else {
            $SuccessMessage = "SUCCESS - All tests passed! Please proceed to creating an Amazon FSx file system. For your convenience, SelfManagedActiveDirectoryConfiguration of result can be used directly in CreateFileSystemWindowsConfiguration for New-FSXFileSystem (see example in README.md)"

            $PwdPointer = [Runtime.InteropServices.Marshal]::SecureStringToBSTR($Context.Credential.Password)
            $Password = [Runtime.InteropServices.Marshal]::PtrToStringAuto($PwdPointer)
            $Context.SelfManagedActiveDirectoryConfiguration = @{
                DnsIps = $Context.DnsIpAddresses
                DomainName = $Context.AdDomain.DNSRoot
                FileSystemAdministratorsGroup = $Context.AdminGroup
                OrganizationalUnitDistinguishedName = $Context.OrganizationalUnit
                Password = $Password
                UserName = $Context.Credential.UserName
            }
        }
        Write-Host $SuccessMessage -ForegroundColor $Context.TestHeaderColor
        Get-PSVersion
    } Else {
        Write-Host "FAILURE - Tests failed. Please see error details below:" -ForegroundColor $Context.TestHeaderColor

        Write-Table ($Context.Failures)

        If ($DomainControllersUnreachable) {
            Write-Host "Not all Domain Controllers are reachable - please refer to the below report for the unreachable Domain Controllers"
            Write-Table ($Context.PingControllersResult.FailedDomainControllers)
        }

        Write-Host "Please address all errors and warnings above prior to re-running validation to confirm fix." -ForegroundColor $Context.TestHeaderColor
    }

    If ((-not $HasNoErrors) -or $HasWarnings) {
        Write-Host "Please refer to the included Troubleshooting Guide (TROUBLESHOOTING.md) for each warning and error." -ForegroundColor $Context.TestHeaderColor
        Get-PSVersion
    }

    If ($TranscriptPath) {
        Stop-Transcript
        Write-Host "Transcript saved to ${TranscriptPath}"
    }

    Return $Context
}
}

function Test-PingControllers {
<#
.SYNOPSIS
    Sends ping requests and tests TCP connectivity of all the specified Domain Controllers.

.DESCRIPTION
    For each Domain Controller this test will:
    1. Send ping requests to all the specified Domain Controllers
    2. Test the TCP connectivity of all Domain Controllers by binding to the following ports: 
       - Port 88   Kerberos authentication
       - Port 389  Lightweight Directory Access Protocol (LDAP)
       - Port 636  Lightweight Directory Access Protocol over TLS/SSL (LDAPS)
       - Port 9389 Microsoft AD DS Web Services, PowerShell
    This test will return both connectivity and latency results of Domain Controllers.

.EXAMPLE
    $DC1 = [PSCustomObject]@{
        Name = 'DC1.test-ad.local'
        IPAddress = '10.0.5.228'
    }
    $DC2 = [PSCustomObject]@{
        Name = 'DC2.test-ad.local'
        IPAddress = '10.0.77.202'
    }
    $Context = @{
        DomainControllers = @($DC1, $DC2)
        Warnings = @{}
        Failures = @{}
        Skip = 0
        Test = 3
        TestHeaderColor = "Green"
    }
    Test-PingControllers -Context $Context
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate all Domain Controllers are reachable"

    If ($Context.Failures.Count -gt 0) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    # Descriptions of the ports we will make sure we can bind to:
    # 66   = "Kerberos authentication"
    # 389  = "Lightweight Directory Access Protocol (LDAP)"
    # 636  = "Lightweight Directory Access Protocol over TLS/SSL (LDAPS)"
    # 9389 = "Microsoft AD DS Web Services, PowerShell"
    $Ports = @(88, 389, 636, 9389)
    $DomainControllers = $Context.DomainControllers
    $TestReport = @()
    $FailedDCs = @()
    $DCCounter = 0

    ForEach ($DC in $DomainControllers) {
        If ($DCCounter -ge $Context.DomainControllersMaxCount) {
            Write-Host "Maximum limit of $($Context.DomainControllersMaxCount) domain controllers have already been checked - skipping the rest..."
            Break
        }

        $CurrentPingResult = [PSCustomObject]@{
            DomainController = "$($DC.Name) ($($DC.IPAddress))"
            PingRTT = "N/A"
            ReachablePorts = @()
            UnreachablePorts = @()
        }

        If (-not ($DC.IPAddress -as [IPAddress] -as [Bool])) {
            Write-Warning "Received a bad IP address: $($DC.IPAddress) for DC: $($DC.Name) - skipping connectivity validation for this record"
            $FailedDCs += [PSCustomObject]@{
                DomainController = $CurrentPingResult.DomainController
                UnreachablePorts = $Ports
            }
            Continue
        }

        # Execute a ping request (does not bind to a port) to the Domain Controller to retrieve RoundTripTime (RTT) values
        $PingResult = Test-NetConnection -ComputerName $DC.IPAddress
        If ($PingResult.PingSucceeded) {
            $CurrentPingResult.PingRTT = "$($PingResult.PingReplyDetails.RoundTripTime)ms"
        }
        
        ForEach ($Port in $Ports) {
            # Validate that we can bind to the port
            $PortConnectionResult = Test-NetConnection -ComputerName $DC.IPAddress -Port $Port 
            If ($PortConnectionResult.TcpTestSucceeded) {
                $CurrentPingResult.ReachablePorts += $Port
            } Else {
                Write-Warning "The Domain Controller $($CurrentPingResult.DomainController) is unreachable on port $($Port)"
                $CurrentPingResult.UnreachablePorts += $Port
            }
        }

        If ($CurrentPingResult.UnreachablePorts.Count -gt 0) {
            $FailedDCs += [PSCustomObject]@{
                DomainController = $CurrentPingResult.DomainController
                UnreachablePorts = $CurrentPingResult.UnreachablePorts
            }
        }

        $TestReport += $CurrentPingResult
        $DCCounter++
    }

    Write-Table $TestReport

    $Context.PingControllersResult = @{
        FailedDomainControllers = $FailedDCs
        StatusReport = $TestReport
    }
    $Context.Test++
    Return $Context
}
}
function Get-PSVersion {
    $ClientPSVersion = (get-host).Version
    Write-Host "The Client Powershell version running this script is-" $ClientPSVersion -ForegroundColor $Context.TestHeaderColor
}

function Test-ComputerObjects {
<#
.SYNOPSIS
    Test AD computer objects are healthy

.DESCRIPTION
    Test AD computer objects exist, are enabled, and are located in the desired OU

.EXAMPLE
    $OrganizationalUnit = 'OU=Amazon FSx,OU=test-ad,DC=test-ad,DC=local'
    $AdDomain = [PSCustomObject]@{
        Name = 'test-ad'
        DNSRoot = 'test-ad.local'
    }
    $ReachableDC = [PSCustomObject]@{
                Name = 'DC1.test-ad.local'
                IPAddress = '10.0.5.228'
            }
    $ComputerObjects = @('amzfsx123','amznfsx456')
    $Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
    $Credential = New-Object System.Management.Automation.PSCredential ("fsxServiceUser", $Password)
    $Context = @{
        OrganizationalUnit = $OrganizationalUnit
        AdDomain = $AdDomain
        ReachableDC = $ReachableDC
        ComputerObjects = $ComputerObjects
        Credential = $Credential
        Failures = @{}
        Warnings = @{}
        Skip = 0
        Test = 18
        TestHeaderColor = "Green"
    }
    Test-ComputerObject -Context $Context
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate existing computer objects"

    If ($Context.Failures.Count -gt 0) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    if (!$Context.ComputerObjects) {
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    $CommonArgs = @{
        Server = $Context.ReachableDC.IPAddress
        Credential = $Context.Credential
    }

    $ErrorActionPreference = 'SilentlyContinue'
    foreach ($ComputerName in $Context.ComputerObjects) {
        $ComputerObject = $null  # reset $ComputerObject
        $ComputerObject = Get-ADComputer -Identity:$ComputerName -Properties:* @CommonArgs
        Write-Table ($ComputerObject | Select Name, DistinguishedName, SamAccountName)

        if (!$ComputerObject) {
            Write-Warning "Computer object $ComputerName could not be found"
            $WarningMessage = "InvalidComputerObject"
            if ($Context.Warnings.ContainsKey($WarningMessage)) {
                $Context.Warnings[$WarningMessage] += $ComputerName
            } else { 
                $Context.Warnings.Add($WarningMessage, @($ComputerName))
            }
        } else {
            
            if ($ComputerObject.DistinguishedName -notlike "*"+$Context.OrganizationalUnit+"*") {
                Write-Warning ("Computer object $ComputerName exists but is not located in " + $Context.OrganizationalUnit)
                $WarningMessage = "MisplacedComputerObject"
                if ($Context.Warnings.ContainsKey($WarningMessage)) {
                    $Context.Warnings[$WarningMessage] += $ComputerName
                } else {
                    $Context.Warnings.Add($WarningMessage, @($ComputerName))
                }
            }
            
            if (!$ComputerObject.enabled) {
                Write-Warning ("Computer object $ComputerName is disabled")
                $WarningMessage = "DisabledComputerObject"
                if ($Context.Warnings.ContainsKey($WarningMessage)) {
                    $Context.Warnings[$WarningMessage] += $ComputerName
                } else {
                    $Context.Warnings.Add($WarningMessage, @($ComputerName))
                }
            }
        }
    }

    $Context.Test++
    Return $Context
}
}


function Test-GlobalCatalog {
<#
.SYNOPSIS
    Test that a global catalog of the forest can be contacted

.DESCRIPTION
    Test that a global catalog of the forest can be contacted

.EXAMPLE
    $DnsIpAddresses = @("10.0.88.17", "10.0.77.202")
    $VPC = [PSCustomObject]@{
            CidrBlock = "10.0.0.0/16"
            VpcId = "vpc-00811ed2298428798"
        }
    $Forest = "forest.local"
    $Context = @{
        DnsIpAddresses = $DnsIpAddresses
        VPC = $VPC
        Forest = $Forest
        Warnings = @{}
        Failures = @{}
        Skip = 0
        Test = 20
        TestHeaderColor = "Green"
    }
    Test-GlobalCatalog -Context $Context

.LINK
    See description of process in https://blogs.msmvps.com/acefekay/category/dc-locator-process
#>
[CmdletBinding()]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [Object]
    $Context
)
Process {
    $TestName = "Validate global catalog"

    If (-not $Context.Forest) {
        Write-Host "Forest name was not found" -ForegroundColor $Context.TestHeaderColor
        Write-Host "Skipping $TestName ..." -ForegroundColor $Context.TestHeaderColor
        $Context.Skip++
        Return $Context
    }

    $Test = $Context.Test
    Write-Host "Test $Test - $TestName ..." -ForegroundColor $Context.TestHeaderColor

    If (($Context.DnsIpAddresses.Count -eq 0) -or ($Context.DnsIpAddresses.Count -gt 2)) {
        Write-Warning "Must pass either one or preferably two unique DNS IP addresses!"
        $Context.Failures.Add("InvalidDnsIpCount", $Context.DnsIpAddresses.Count)
        $Context.Test++
        Return $Context
    }

    $SrvRecord = "_ldap._tcp.gc._msdcs." + $Context.Forest

    $Result = (Resolve-FSxDcDnsRecord -SrvRecord $SrvRecord -DnsIpAddresses $Context.DnsIpAddresses -VpcCIDR $Context.VPC.CidrBlock)

    If ($Result.Failures.Count -gt 0) {
        Copy-Entries $Result.Failures $Context.Failures
        Write-Warning "Failed to resolve global catalog DNS record for forest " + $Context.Forest
    } Else {
        $Context.GlobalCatalogs = $Result.DomainControllers
    }

    If ($Result.Warnings.Count -gt 0) {
        Copy-Entries $Result.Warnings $Context.Warnings
    }

    $ReachableDCs = @()
    ForEach ($DC in $Context.GlobalCatalogs) {
        If (Assert-IpIsFSxAccessible -Ip $DC.IPAddress -VpcCIDR $Context.VPC.CidrBlock) {
            Try {
                $CommonAdArgs = @{
                    Server = $DC.IPAddress
                    Credential = $Credential
                    ErrorAction = 'SilentlyContinue'
                }
                $AdDomain = (Get-AdDomain @CommonAdArgs)
                $ReachableDCs += $DC
                break
            }
            Catch [System.Security.Authentication.AuthenticationException] {
                Write-Warning "Invalid credentials provided!"
            }
            Catch [Microsoft.ActiveDirectory.Management.ADServerDownException] {
                # Unable to contact the server. This may be because this server does not exist,
                # it is currently down, or it does not have the Active Directory Web Services running.
                Write-Warning($_)
            }
            Catch {
                Write-Warning("Failed to connect due to unknown exception. " + $_)
            }


        } Else {
            Write-Warning("Unable to communicate with the following AD Domain Controller because its IP is outside the file system's VPC and is not an RFC1918 IP address: " + $DC.Name + " (" + $DC.IPAddress + ")")
        }
    }

    If ($ReachableDCs) {
        $FirstReachableDC = $ReachableDCs[0]
        $ReachableDCIp = $FirstReachableDC.IPAddress
        Write-Host "Running full network validation against $ReachableDCIp confirm all ports are open"
        $NetworkValidationResult = Test-FSxADControllerConnection -ADControllerIp $ReachableDCIp
        If ($NetworkValidationResult.Success) {
            Write-Host "All ports on $ReachableDCIp are open"
        } Else {
            Write-Host "Some ports on $ReachableDCIp are closed. Please check connectivity."
            $Warnings.Add('DomainControllerNetworkValidation', $NetworkValidationResult)
        }
    }

    $Context.Test++

    Return $Context
}
}
