# Amazon FSx Active Directory Validation PowerShell Module

## Introduction

This module provides Amazon FSx for Windows File Server customers with an easy way to validate that their Active Directory (AD) configuration is ready for integration with Amazon FSx.

This module helps to automate checks that all the [Self-Managed Active Directory Prerequisites](https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-manage-prereqs.html) are met.
This automates validation of your AD settings such as security groups, network ACLs, site configuration, service account permissions, etc.

This module contains the following PowerShell commandlets:

- `Test-FSxADConfiguration` - end-to-end validation script
- `Test-FSxADControllerConnection` - test network connectivity to domain controller
- `Test-FSxADDnsConnection` - test network connectivity to DNS server

These are described in their respective sections below.
`Get-Help <command> -detailed` can also provide detailed help with
parameter description and usage examples.

A bonus standalone script `Add-FSxADPermissions.ps1` is provided
to help [delegate service account permissions](https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-managed-AD-best-practices.html#connect_delegate_privileges). Usage description is provided in Appendix A of `TROUBLESHOOTING.md`.

## Prerequisites

### EC2 Instance Permissions

Run all of the following from a Windows EC2 instance launched in the same
subnet as the one you intend to launch your Amazon FSx file system in.
In case of Multi-AZ deployment types, use the subnet for the preferred
active file server. Please apply the same security groups that you'd like
to apply to the file server.

Please make sure the instance has the following instance role permissions:

- `ec2:DescribeSubnets`
- `ec2:DescribeVpcs`

`AmazonEC2ReadOnlyAccess` managed IAM policy has the required permissions.
You can validate instance role permissions via [IAM policy simulator](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_testing-policies.html)

(Optional) If the validation is successful, we offer an expedited option
to create a new FSx file system directly with parameters from the validation
pre-filled. If you'd like to take advantage of this option, please grant
`fsx:CreateFileSystem` permission to the instance role as well.
The `AmazonFSxFullAccess` managed IAM policy has all of the permissions
mentioned above.

### Active Directory PowerShell Module

Active Directory module for Windows PowerShell must be installed. To test this, try importing it.

```
Import-Module ActiveDirectory
```

If above returns an error, install it

```
Install-WindowsFeature RSAT-AD-PowerShell
```

### AWS Tools for Windows PowerShell EC2

Check version of AWS PowerShell installed (this is generally pre-installed
on Windows EC2 instances).

```
PS C:\Users\peak> Get-AWSPowerShellVersion -ListServiceVersionInfo | Where-Object { $_.'Noun Prefix' -eq 'EC2' }

Service                            Noun Prefix SDK Assembly Version
-------                            ----------- --------------------
Amazon Elastic Compute Cloud (EC2) EC2         3.3.168.0
```

If above returns an error, follow [installation instructions](https://docs.aws.amazon.com/powershell/latest/userguide/pstools-getting-set-up-windows.html)

## Active Directory Validation (Test-FSxADConfiguration)

This is the most end-to-end validation. It's recommended to try this one first
for most users.

### Command Invocation

Import module

```
Import-Module .\AmazonFSxADValidation
```

Set required parameters

```
# Credential for Amazon FSx service account
$Password = ConvertTo-SecureString 'MySecretPassword' -AsPlainText -Force
$Credential = New-Object System.Management.Automation.PSCredential ('fsxServiceUser', $Password)

$FSxADValidationArgs = @{
    # DNS root of Active Directory domain
    DomainDNSRoot = 'test-ad.local'

    # IP v4 addresses of DNS servers
    DnsIpAddresses = @('10.0.88.17', '10.0.77.202')

    # Subnet IDs for Amazon FSx file server(s)
    SubnetIds = @('subnet-04431191671ac0d19', 'subnet-0f30db1cad3a599d1')

    Credential = $Credential
}
```

(Optional) Set organizational unit and admin group if needed

```
$FSxADValidationArgs['OrganizationalUnit'] = 'OU=Amazon FSx,OU=test-ad,DC=test-ad,DC=local'
$FSxADValidationArgs['AdminGroup'] = 'FSxDelegatedAdmins'
```

(Optional) enable service account permission validation.
In order to be able to validate these permissions, this script will
create test Active Directory computer objects in the organizational unit.
This will be cleaned up by the script unless delete permissions are
not properly configured on the provided service account in which case
manual cleanup may be necessary. It is HIGHLY recommended to enable
this validation.

```
$FSxADValidationArgs['TestServiceAccountPermissions'] = $True
```

(Optional) Select directory to record output to as a PowerShell transcript.
If not specified, no transcript will be generated. This can be useful for
debugging.

```
$FSxADValidationArgs['TranscriptDirectory'] = [Environment]::GetFolderPath("MyDocuments")
```

(Optional) Set the maximum number of domain controllers for which we will test connectivity.
In the interest of keeping the test time reasonably short for large AD setups, this parameter will limit the amount of domain controllers for which we will execute connectivity tests.
If not specified, the default value for this parameter will be set to 100.

```
$FSxADValidationArgs['DomainControllersMaxCount'] = 200
```

Run

```
$Result = Test-FSxADConfiguration @FSxADValidationArgs
```

### Example of Complete Success

```
Test 1 - Validate EC2 Subnets ...
...
Test 17 - Validate 'Delete Computer Objects' permission ...

Test computer object amznfsxtestd53f deleted!
...
SUCCESS - All tests passed! Please proceed to creating an Amazon FSx file system. For your convenience, SelfManagedActiveDirectoryConfiguration of result can be used directly in CreateFileSystemWindowsConfiguration for New-FSXFileSystem
PS C:\AmazonFSxADValidation> $Result.Failures.Count
0
PS C:\AmazonFSxADValidation> $Result.Warnings.Count
0
```

If entirely successful with no failures or warnings, please proceed
to create an Amazon FSx file system.

Example create call

[New-FSXFileSystem docs](https://docs.aws.amazon.com/powershell/latest/reference/items/New-FSXFileSystem.html)
[CreateFileSystemWindowsConfiguration docs](https://docs.aws.amazon.com/sdkfornet/v3/apidocs/index.html?page=FSx/TFSxCreateFileSystemWindowsConfiguration.html&tocid=Amazon_FSx_Model_CreateFileSystemWindowsConfiguration)

```
# Fetch security group IDs from current running instance
$FirstMacAddress = ((Get-EC2InstanceMetadata -Path '/network/interfaces/macs') | Select-Object -First 1)
$SecurityGroupIds = (Get-EC2InstanceMetadata -Path "/network/interfaces/macs/${FirstMacAddress}security-group-ids")

$WindowsConfiguration = @{
    ThroughputCapacity = 16
    SelfManagedActiveDirectoryConfiguration = $Result.SelfManagedActiveDirectoryConfiguration
    PreferredSubnetId = $FSxADValidationArgs.SubnetIds[0]
    DeploymentType = 'MULTI_AZ_1'
}

$CreateArgs = @{
    FileSystemType = 'WINDOWS'
    StorageCapacity = 300
    SubnetId = $FSxADValidationArgs.SubnetIds
    SecurityGroupIds = $SecurityGroupIds
    WindowsConfiguration = $WindowsConfiguration
}

$FileSystem = New-FSXFileSystem @CreateArgs
```

### Example of Success with Warnings

```
Test 1 - Validate EC2 Subnets ...
...
Test 2 - Validate connectivity with DNS Servers ...
WARNING: Unable to communicate with the following DNS Server because its IP is outside the file system's VPC and is not
 an RFC1918 IP address: 96.127.0.0
...
Test 8 - Looking up DNS entries for domain controllers in site SiteB ...
WARNING: Unable to communicate with the following DNS Server because its IP is outside the file system's VPC and is not
 an RFC1918 IP address: 96.127.0.0
...
Test 17 - Validate 'Delete Computer Objects' permission ...

Test computer object amznfsxtest2358 deleted!
...
Possible configuration issues identified. Please see warning details below:

Name                           Value
----                           -----
DnsCommunication               {96.127.0.0}



SUCCESS with WARNINGS - All tests passed but there were possible configuration issues. It is HIGHLY recommended to address these before creating an Amazon FSx file system.
PS C:\AmazonFSxADValidation> $Result.Failures.Count
0
PS C:\AmazonFSxADValidation> $Result.Warnings.Count
1
PS C:\AmazonFSxADValidation> $Result.Warnings

Name                           Value
----                           -----
DnsCommunication               {96.127.0.0}
```

If successful but there are warnings, it is HIGHLY recommended
to resolve the warnings and rerun validation
before creating an Amazon FSx file system.

In the illustrated case, FSx is unable to talk to one of two
DNS servers provided. Availability is reduced to that of a single
reachable DNS server rather than two as intended.

For detailed information about a given warning, see `TROUBLESHOOTING.md`.

### Example of Failures

```
Test 1 - Validate EC2 Subnets ...
...
Test 7 - Validate that provided EC2 Subnets belong to a single AD Site ...

Name          DistinguishedName                                                         Site
----          -----------------                                                         ----
10.0.0.0/19   CN=10.0.0.0/19,CN=Subnets,CN=Sites,CN=Configuration,DC=test-ad,DC=local   CN=SiteB,CN=Sites,CN=Configu...
10.0.128.0/19 CN=10.0.128.0/19,CN=Subnets,CN=Sites,CN=Configuration,DC=test-ad,DC=local CN=Default-First-Site-Name,C...
10.0.64.0/19  CN=10.0.64.0/19,CN=Subnets,CN=Sites,CN=Configuration,DC=test-ad,DC=local  CN=SiteB,CN=Sites,CN=Configu...



Best match for EC2 subnet subnet-092f4caca69e360e7 is AD site CN=Default-First-Site-Name,CN=Sites,CN=Configuration,DC=te
st-ad,DC=local
Best match for EC2 subnet subnet-04431191671ac0d19 is AD site CN=SiteB,CN=Sites,CN=Configuration,DC=test-ad,DC=local
WARNING: EC2 subnets subnet-092f4caca69e360e7 subnet-04431191671ac0d19 matched to different AD sites! Make sure they
are in a single AD site.
...
9 of 17 tests skipped.
FAILURE - Tests failed. Please see error details below:

Name                           Value
----                           -----
SubnetsInSeparateAdSites       {subnet-04431191671ac0d19, subnet-092f4caca69e360e7}



Please address all errors and warnings above prior to re-running validation to confirm fix.
PS C:\AmazonFSxADValidation> $Result.Failures.Count
1
PS C:\AmazonFSxADValidation> $Result.Failures

Name                           Value
----                           -----
SubnetsInSeparateAdSites       {subnet-04431191671ac0d19, subnet-092f4caca69e360e7}


PS C:\AmazonFSxADValidation> $Result.Warnings.Count
0
```

Please work through ALL failures and rerun validation to confirm fix
before attempting to create an Amazon FSx file system.
Reruns are necessary because failures cause later tests to be skipped.

In the illustrated case, the EC2 subnets for Amazon FSx integration are
mapped to Active Directory subnets in different sites. This can cause
high latency due to inter-site replication delay. It is recommended to map
the subnets to a single AD site.

For detailed information about a given failure, see `TROUBLESHOOTING.md`.

## Domain Controller Network Connectivity (Test-FSxADControllerConnection)

End-to-end validation tool will not run full network connectivity check against
every domain controller in the domain. Use this if you would like to run
network connectivity validation against specific set of domain controllers.

### Command Invocation

Import module

```
Import-Module .\AmazonFSxADValidation
```

Set variables and run

```
$ADControllerIp = <e.g. '10.0.75.243'>
$Result = Test-FSxADControllerConnection -ADControllerIp $ADControllerIp
```

### Sample Success Output

```
PS C:\AmazonFSxADValidation> $Result

Name                           Value
----                           -----
TcpDetails                     {@{Port=88; Result=Listening; Description=Kerberos authentication}, @{Port=135; Resul...
Server                         10.0.75.243
UdpDetails                     {@{Port=88; Result=Timed Out; Description=Kerberos authentication}, @{Port=123; Resul...
Success                        True


PS C:\AmazonFSxADValidation> $Result.TcpDetails

Port Result    Description
---- ------    -----------
  88 Listening Kerberos authentication
 135 Listening DCE / EPMAP (End Point Mapper)
 389 Listening Lightweight Directory Access Protocol (LDAP)
 445 Listening Directory Services SMB file sharing
 464 Listening Kerberos Change/Set password
 636 Listening Lightweight Directory Access Protocol over TLS/SSL (LDAPS)
3268 Listening Microsoft Global Catalog
3269 Listening Microsoft Global Catalog over SSL
9389 Listening Microsoft AD DS Web Services, PowerShell
```

### Sample Failure Output

```
PS C:\> $Result = Test-FSxADControllerConnection -ADControllerIp $ADControllerIp
WARNING: TCP 9389 failed to connect. Required for Microsoft AD DS Web Services, PowerShell. 
Verify security group and firewall settings on both client and directory controller.
WARNING: 1 ports failed to connect to 10.0.75.243. Check pre-requisites in
https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-managed-AD.html#self-manage-prereqs

PS C:\AmazonFSxADValidation> $Result

Name                           Value
----                           -----
TcpDetails                     {@{Port=88; Result=Listening; Description=Kerberos authentication}, @{Port=135; Resul...
Server                         10.0.75.243
UdpDetails                     {@{Port=88; Result=Timed Out; Description=Kerberos authentication}, @{Port=123; Resul...
Success                        False
FailedTcpPorts                 {9389}


PS C:\AmazonFSxADValidation> $Result.FailedTcpPorts
9389
```

Windows socket error code mapping

https://msdn.microsoft.com/en-us/library/ms740668.aspx

#### Additional resources

For more detailed network debugging on Microsoft platforms,
consider using PortQuery

[Microsoft blog for testing domain controller connectivity](https://blogs.msdn.microsoft.com/muaddib/2009/03/29/testing-domain-controller-connectivity-using-portqry)

[Port query description](https://support.microsoft.com/en-us/help/310099/description-of-the-portqry-exe-command-line-utility)

[Port query v2 download](https://www.microsoft.com/en-us/download/details.aspx?id=17148)

## DNS Server Network Connectivity (Test-FSxADDnsConnection)

Set variables and run

```
$ADDnsIp = <e.g. '10.0.55.102'>
$Result = Test-FSxADDnsConnection -ADDnsIp $ADDnsIp
```

Output is similar structure as the domain controller test above.

## Questions?

In case the above validation and `TROUBLESHOOTING.md`
was insufficient to pinpoint the issue, please
contact us through [AWS Support](https://aws.amazon.com/support)
and include the transcript from the latest run of the Active Directory validation script.
