﻿# Copyright 2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.


#
# Restore volume to an EC2 instance 
# WARNING: This will stop the instance and dismount existing volumes
#
function Vss-Restore {
    param(
        [String]$InstanceId,
        [String[]]$Snapshots
    )

    #get snapshot information
    $SnapshotData = @()
    foreach ($Snapshot in $Snapshots) {
        #Get device from snapshot 'Device' tag
        $filter = New-Object Amazon.EC2.Model.Filter
        $filter.Name = "key"
        $filter.Values = "Device"
        $filters = @($filter)
        $keys = new-object string[] 1
        $keys[0] = $Snapshot
        $filter = new-object Amazon.EC2.Model.Filter
        $filter.Name = 'resource-id'
        $filter.Values = $keys
        $filters += @($filter)
        $DeviceTag = Get-EC2Tag -Filter $filters
        if ($DeviceTag -eq $null) {
            throw "Restore failed, snapshot $Snapshot missing 'Device' tag"
        }

        $SnapshotData +=
        New-Object PSObject -Property @{
            EbsVolumeId = $null
            Device      = $DeviceTag.Value
            SnapshotId  = $Snapshot
        }   
    }


    Write-Host "Shutting down instance" $InstanceId
    Stop-EC2Instance -Instance $InstanceId > $null
    do {
        Sleep -Seconds 2
        $Instance = (Get-EC2Instance -Instance $InstanceId).Instances
    }
    until($Instance.State.Name -eq "stopped")

    #Dismount existing volumes from instance, ;eaving root volume if not replacing it
    foreach ($BlockDevice in $Instance.BlockDeviceMappings) {
        if ($SnapshotData.Device.Contains($BlockDevice.DeviceName) -or $BlockDevice.DeviceName -ne "/dev/sda1") {
            Write-Host "Dimounting volume" $BlockDevice.Ebs.VolumeId "for device" $BlockDevice.DeviceName "from instance" $InstanceId
            Dismount-EC2Volume -InstanceId $InstanceId -VolumeId $BlockDevice.Ebs.VolumeId > $null
        }
    }
    
    #create volumes from snapshot and attach them to instance
    $AvailabilityZone = $Instance.Placement.AvailabilityZone
    foreach ($Data in $SnapshotData) {

        #TODO: Modify to set iops, encryption, etc
        $Volume = New-EC2Volume -SnapshotId $Data.SnapshotId -AvailabilityZone $AvailabilityZone
        
        do {
            Sleep -Seconds 5
            $VolumeState = (Get-EC2Volume -VolumeId $Volume.VolumeId).State
        }
        until($VolumeState = "available")
        $Data.EbsVolumeId = $Volume.VolumeId

        Write-Host "Attaching volume" $Data.EbsVolumeId "from snapshot" $Data.SnapshotId "for device" $Data.Device "to instance" $InstanceId
        Add-EC2Volume -InstanceId $InstanceId -VolumeId $Volume.VolumeId -Device $DeviceTag.Value > $null
    }

    #Start instance
    Write-Host "Starting instance" $InstanceId
    Start-EC2Instance -InstanceId $InstanceId > $null
    
}


Vss-Restore $InstanceId @($SnapshotId)

